const ordersSignature = function (list) {
    if (!Array.isArray(list)) {
        return '[]';
    }

    try {
        return JSON.stringify(list);
    } catch (e) {
        return String(list.length || 0);
    }
};

Vue.component('patientOrders', {

    props: [
        'routes',
        'events',
        'initialOrders'
    ],

    data: function() {
        const initialProvided = Array.isArray(this.initialOrders);
        const initial = initialProvided ? this.initialOrders.slice() : [];
        return {
            orders: initial,
            hasLoaded: initialProvided,
            loading: false,
            errors: {},
            success: false,
            display: 3,
            eventIndex: 0,
            builderNonce: 0,
            bootstrapOrdersSignature: ordersSignature(initial),
            ordersOverlayBound: false,

        };
    },

    watch: {
        initialOrders: {
            immediate: true,
            handler(list) {
                const next = Array.isArray(list) ? list : [];
                const signature = ordersSignature(next);

                if (signature === this.bootstrapOrdersSignature) {
                    return;
                }

                this.bootstrapOrdersSignature = signature;
                this.orders = next.slice();
                this.hasLoaded = true;
                this.loading = false;
                this.notifyOrdersLoaded();
                this.bindModals();
            }
        },
        events(stack) {
            // Because there may be more than one event at a time, loop through new events
            for (let i = this.eventIndex; i < stack.length; i++) {
                let event = stack[i];
                if (event.name === 'orderCreated') {
                    this.load(true);
                }
            }
            // Update event index so that we don't process already processed events
            this.eventIndex = stack.length;
        }
    },

    mounted: function() {
        this.init();
    },

    computed: {
        canLoadMore() {
            const list = Array.isArray(this.orders) ? this.orders : [];
            return list.length > this.display;
        }
    },

    methods: {
        // Run BEFORE the overlay's click handler so the builder is ready
        // (This gets called from a mousedown handler / or delegated listener)
        prepBuilder(orderId) {
            // clear any stale UI state
            this.success = false;
            this.errors  = {};
            // tell the Basket Builder to reset itself for a new session
            this.emitEvent('initBasketBuilder', {
                orderId: orderId > 0 ? orderId : null,
                fresh: true,
            });
        },
        init() {
            if (this.hasLoaded) {
                this.notifyOrdersLoaded();
                this.bindModals();
                return;
            }
            this.load(true);
        },
        load(force = false) {
            const vm = this;
            if (!force && this.hasLoaded) {
                return;
            }

            const applyPayload = (payload) => {
                const list = Array.isArray(payload?.orders) ? payload.orders : [];
                const signature = ordersSignature(list);
                const changed = signature !== vm.bootstrapOrdersSignature;

                vm.orders = list.slice();
                vm.hasLoaded = true;
                vm.loading = false;

                if (changed) {
                    vm.bootstrapOrdersSignature = signature;
                }

                vm.notifyOrdersLoaded(changed);

                vm.bindModals();

                if (changed && typeof vm.cacheBootstrap === 'function') {
                    vm.cacheBootstrap('admin-member-orders-load', { orders: list.slice() });
                }
            };

            const cached = (!force && this.consumeBootstrap)
                ? this.consumeBootstrap('admin-member-orders-load')
                : null;
            if (cached) {
                applyPayload(cached);
                return;
            }

            this.loading = true;
            this.$http.post(this.routes['admin-member-orders-load'], {})
                .then((response) => {
                    let payload = response.data;
                    if (typeof payload === 'string') {
                        try {
                            payload = JSON.parse(payload);
                        } catch (e) {
                            console.error('Invalid JSON from admin-member-orders-load:', e, payload);
                            payload = {};
                        }
                    }

                    applyPayload(payload);
                })
                .catch((err) => {
                    console.error('Orders load failed:', err);
                    vm.loading = false;
                    vm.orders = [];
                    vm.errors.load = 'Failed to load orders.';
                });
        },
        addOrderAction(orderId) {
            this.emitEvent('initBasketBuilder', {
                orderId: orderId > 0 ? orderId : null,
            });
            return true;
        },
        loadMore() {
            this.display += 10;
            this.notifyOrdersLoaded(true);
            this.bindModals();
        },
        addNoteAction(id) {
            this.emitEvent('initAddNote', {
                orderID: id,
            })
        },
        emitEvent(name, data) {
            this.$parent.emitEvent(name, data);
        },
        notifyOrdersLoaded(force = false) {
            if (!this.ordersOverlayBound || force) {
                this.ordersOverlayBound = true;
                this.emitEvent('ordersLoaded');
            }
        },
        bindModals() {
            const vm = this;
            this.$nextTick(function () {
                // Prime the Basket Builder BEFORE the overlay clones/opens (#basketBuilder)
                // Do it on mousedown so it runs earlier than click.
                $(document)
                    .off('mousedown.patientOrders')
                    .on('mousedown.patientOrders', '.addOrderButton', function () {
                        const raw = $(this).attr('data-order-id'); // "0" or a number
                        const id  = raw == null ? 0 : parseInt(raw, 10);
                        const parentId = (Number.isFinite(id) && id > 0) ? id : null;

                        vm.builderNonce = (vm.builderNonce + 1) % Number.MAX_SAFE_INTEGER;
                        vm.emitEvent('initBasketBuilder', {
                            orderId: parentId,   // null for top-level; number for sub-order
                            fresh: true,
                            nonce: vm.builderNonce,
                        });
                    });

                // Ensure the add button actually opens the overlay
                try { $('.addOrderButton').mtcOverlay({}); } catch (e) {}

                // Existing "view order" overlay binding
                try {
                    $('.viewOrderButton').mtcOverlay({
                        onClose() {},
                        onOpen() {}
                    });
                } catch (e) {}
                // ---------------- Add Patient Note overlay ----------------
                $(document)
                    .off('click.patientNotes')
                    .on('click.patientNotes', 'a.addPatientNoteButton', function (e) {
                        e.preventDefault();

                        const $trigger = $(this);
                        const orderId  = $trigger.data('order-id');
                        const $root    = $('#addPatientNote');

                        if ($root.length) {
                            // set order id every time
                            $root.find('input[name="order_id"]').val(orderId || '');
                            // clear textarea + any stale editor
                            const $ta = $root.find('textarea#patientNotesInput.simpletinymce');
                            if ($ta.length) {
                                if (window.tinymce) {
                                    const ed = tinymce.get('patientNotesInput');
                                    if (ed) ed.remove();
                                }
                                $ta.val('');
                            }
                        }

                        // open overlay
                        try { $trigger.mtcOverlay('open'); }
                        catch (_) { try { $trigger.mtcOverlay({}).mtcOverlay('open'); } catch (__) {} }
                    });

// init/destroy TinyMCE when the overlay opens/closes
                try {
                    $('a.addPatientNoteButton').mtcOverlay({
                        onOpen() {
                            const boot = function () {
                                if (typeof init_tinymce === 'function') {
                                    if (window.tinymce) {
                                        const ed = tinymce.get('patientNotesInput');
                                        if (ed) ed.remove(); // reset
                                    }
                                    init_tinymce();

                                    // Keep TinyMCE and v-model in sync
                                    setTimeout(function () {
                                        const ed = window.tinymce && tinymce.get('patientNotesInput');
                                        if (!ed) return;
                                        const sync = () => {
                                            ed.save(); // iframe -> textarea
                                            const ta = document.getElementById('patientNotesInput');
                                            if (ta) {
                                                ta.dispatchEvent(new Event('input', { bubbles: true }));
                                                const legacy = ta.form && ta.form.querySelector('input[name="note"]');
                                                if (legacy) legacy.value = ta.value;
                                            }
                                        };
                                        ed.on('change keyup SetContent', sync);
                                    }, 0);
                                }
                            };
                            setTimeout(boot, 0);
                        },
                        onClose() {
                            if (window.tinymce) {
                                const ed = tinymce.get('patientNotesInput');
                                if (ed) ed.remove();
                            }
                        }
                    });
                } catch (e) {}
// -----------------------------------------------------------


            });
        }
    },
});
