import {defineConfig, loadEnv} from 'vite';
import vue2 from '@vitejs/plugin-vue2';
import {resolve} from 'path';
import stylehacks from 'stylehacks';

// ... existing code ...
export default defineConfig(({mode}) => {
    const env = loadEnv(mode, process.cwd(), '');

    if (!env.APP_DOMAIN) {
        console.error('Missing APP_DOMAIN in your .env file. Please set APP_DOMAIN to your domain.');
        process.exit(1);
    }

    // if we are in the staging environment we need the SSL certs to be added or we will get a cors error especially when on a fork / subdomain
    if ((!env.SSL_KEY_PATH || !env.SSL_CERT_PATH) && env.APP_ENV === 'staging') {
        console.error('Missing SSL_KEY_PATH or SSL_CERT_PATH. See VITE-MIGRATION.md.');
        process.exit(1);
    }

    const APP_DOMAIN = env.APP_DOMAIN;
    const APP_URL = env.APP_URL || `https://${APP_DOMAIN}`;

    return {
        plugins: [
            vue2()
        ],
        server: {
            host: '0.0.0.0',
            port: 5173,
            https: env.SSL_KEY_PATH && env.SSL_CERT_PATH
                ? {key: env.SSL_KEY_PATH, cert: env.SSL_CERT_PATH}
                : false,
            cors: true,
            allowedHosts: [
                APP_DOMAIN,
                `.${APP_DOMAIN}`,
                'localhost'
            ]
        },
        optimizeDeps: {
            include: ['swiper', 'swiper/vue'],
            force: true,
            esbuildOptions: {
                resolveExtensions: ['.js', '.vue']
            }
        },
        build: {
            outDir: 'public/build',
            emptyOutDir: true,
            manifest: true,
            rollupOptions: {
                input: {
                    app: resolve(__dirname, 'sites/default/js/app.js'),
                    style: resolve(__dirname, 'sites/default/css/app.less')
                }
            }
        },
        resolve: {
            alias: {
                '@site': resolve(__dirname, 'sites/default'),
                '@core': resolve(__dirname, 'core'),
                '@components': resolve(__dirname, 'sites/default/js/components'),
                '@plugins': resolve(__dirname, 'plugins'),
                '@shop': resolve(__dirname, 'shop'),
                '@site_scripts': resolve(__dirname, 'sites/default/js/site_scripts'),
                '@mtcmedia': resolve(__dirname, 'vendor/mtcmedia/'),
                // Add this line to use the full Vue build
                'vue': 'vue/dist/vue.esm.js'
            },
            extensions: ['.mjs', '.js', '.ts', '.jsx', '.tsx', '.json', '.vue'],
        },
        css: {
            postcss: {
                plugins: [
                    stylehacks()
                ]
            },
            preprocessorOptions: {
                less: {
                    javascriptEnabled: true,
                    additionalData: `
                        @import "@site/css/variables.less";
                        @import "@core/css/mixins.less";
                        @import "@site/css/base/mixins.less";
                      `
                }
            }
        },
        publicDir: false,
        base: '/public/build/'
    };
});
// ... existing code ...