<?php
/**
 * Item Size Object Eloquent model.
 * @version 21/09/16
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Shop\Item;

use Illuminate\Database\Eloquent\Model;
use Mtc\Shop\Assessment\Form;
use Mtc\Shop\ComparisonSiteItem;
use Mtc\Shop\Ingredient;

/**
 * Item Size Object Eloquent model.
 * Implements functionality for managing Shop Item Sizes
 *
 * @version 21/09/16
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class Size extends Model
{
    // Use trait that defines relationship to items
    use BelongsToItem;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'items_sizes';

    /**
     * @var array The attributes that are mass assignable.
     */
    protected $fillable = [
        'item_id',
        'size',
        'label_count',
        'quantity',
        'pack_size',
        'strength',
        'price',
        'price_exvat',
        'sale_price',
        'sale_price_exvat',
        'stock',
        'order',
        'manual',
        'PLU',
        'ROOT_PLU',
        'ean',
        'hide',
        'epos_info',
        'form_id',
    ];

    /**
     * Scope - active()
     * Discards all hidden sizes
     *
     * @param \Illuminate\Database\Eloquent\Builder $query Query builder object
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeActive($query)
    {
        return $query->where('hide', 0);
    }

    /**
     * Scope - selectBasics()
     * Retrieves basic info about size - id, size, colour, price, sale_price and stock
     *
     * @param \Illuminate\Database\Eloquent\Builder $query Query builder object
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeSelectBasics($query)
    {
        return $query->select([
            'id',
            'size',
            'hide',
            'strength',
            'quantity',
            'price',
            'sale_price',
            'stock'
        ]);
    }

    /**
     * Define the relationship to Ingredient
     * @return \Illuminate\Database\Eloquent\Relations\belongsToMany
     */
    public function ingredients()
    {
        return $this->belongsToMany(Ingredient::class, 'items_ingredients', 'size_id');
    }

    /**
     * Define the relationship to Custom fields
     * @return \Illuminate\Database\Eloquent\Relations\hasOne
     */
    public function custom()
    {
        return $this->hasOne(SizeCustom::class);
    }

    /**
     * Finds item assessment form first by item, then by category
     *
     * @return \Illuminate\Database\Eloquent\Builder|\Illuminate\Database\Eloquent\Builder[]|\Illuminate\Database\Eloquent\Collection|Model
     */
    public function findAssessmentForm()
    {
        return $this->form_id ? Form::query()->find($this->form_id) : $this->item->findAssessmentForm();
    }


    public function getAdminUrl()
    {
        return "/shop/admin/products/manage.sizes.php?id={$this->item->id}&sizeid={$this->id}";
    }


    public function isInStock() : bool
    {
        return (bool)$this->stock;
    }
}
