<?php
/**
 * @class GridConstructorController
 * @author Nithika Senaweera <nithika.senaweera@mtc.co.uk>
 * @copyright 2025 mtc. http://www.mtcmedia.co.uk/
 * @version 2.0
 * @description GridContructorController class handles the grid constructor functionality in the GridManager plugin's
 *              admin section.
 * */

namespace Mtc\Plugins\GridManager\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Mtc\Plugins\GridManager\Models\Grid;
use Mtc\Plugins\GridManager\Models\GridStructures;


class GridConstructorController extends Controller
{
    protected $admin_user;
    // constructor
    public function __construct()
    {
       $this->admin_user = new \AdminUser();
    }
    /**
     * Display the grid constructor page.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(): \Illuminate\Http\JsonResponse
    {
        // return a JSON response with the grid
        return response()->json([
            'status' => 'success',
            'message' => 'Grid constructor loaded successfully',
        ], 200);
    }
    /**
     * @description Get the grid structure for a given parent ID.
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @author Nithika Senaweera <nithika.senaweera@mtc.co.uk>
     * @version 2.0
     * @copyright MTC 2025. https://mtcmedia.co.uk
     * */
    public function get(Request $request): \Illuminate\Http\JsonResponse
    {

        // extract the request parameters
        $parent_id = $request->input('parentId');


         //existing structure
        $grid_structure = GridStructures::where('parent_id', $parent_id)->first();

        if ($grid_structure) {
            $json = json_decode($grid_structure->json, true);
            return response()->json([
                'status' => 'success',
                'structure' => $json,
            ], 200);
        }
        else {
            return response()->json([
                'status' => 'error',
                'message' => 'Grid structure not found',
            ], 200);
        }
    }
    /**
     * @description Save the grid structure.
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @author Nithika Senaweera <nithika.senaweera@mtc.co.uk>
     * @version 2.0
     * @copyright MTC 2025. <https://mtcmedia.co.uk>
     * */
    public function save(Request $request): \Illuminate\Http\JsonResponse
    {
        // Validate the request data
        $request->validate([
            'structure' => 'required|json',
            'parentId' => 'required|integer|min:1',
            'type' => 'required|string|in:cms,category,product'
        ]);

        if ($request->accepts(['application/json'])) {

            $user = $this->admin_user->user['username'];
            // extract the request parameters
            $structure = $request->input('structure');
            $parent_id = $request->input('parentId');
            $type = $request->input('type');

            // get the grid by page ID
            $grid = Grid::where('page_id', $parent_id)->first();

            // existing structure
            $grid_structure = GridStructures::where('parent_id', $parent_id)->first();

            if ($grid_structure) {
                $grid_structure->json = $structure;
                $grid_structure->parent_id = (int)$parent_id;
                $grid_structure->type = (string)$type;
                $grid_structure->grid_id = $grid->id;
                $grid_structure->last_modified_by = $user;
                $grid_structure->ip_address = $request->ip();

                // Grid save structures
                $grid_structure->save();
            }
            else {
                // Grid Constructor model
                $model = new GridStructures();

                $model->json = $structure;
                $model->parent_id = (int)$parent_id;
                $model->type = (string)$type;
                $model->grid_id = (int)$grid->id;
                $model->last_modified_by = $user;
                $model->ip_address = $request->ip();

                // Grid save structures
                $model->save();
            }
            return response()->json([
                'status' => 'success',
                'message' => 'Page Saved successfully',
            ], 200);
        }
        else {
            // return an error response if the request does not accept JSON
            return response()->json([
                'status'  => 'error',
                'message' => 'Invalid request format. Please use JSON.',
            ], 400);
        }
    }
    /**
     * @description delete the grid structure.
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @version 2.0
     * @author Nithika Senaweera <nithika.senaweera@mtc.co.uk>
     * @copyright MTC 2025. https://mtcmedia.co.uk
     *
     * */
    public function delete(Request $request): \Illuminate\Http\JsonResponse
    {
        // extract the request parameters
        $parent_id = $request->input('parentId');

        // find the grid structure by parent ID
        $grid_structure = GridStructures::where('parent_id', $parent_id)->first();

        if ($grid_structure) {
            // delete the grid structure
            $grid_structure->delete();
            return response()->json([
                'status' => 'success',
                'message' => 'Grid structure deleted successfully',
            ], 200);
        }
        else {
            return response()->json([
                'status' => 'error',
                'message' => 'Grid structure not found',
            ], 404);
        }
    }

    /**
     * @description Get whether the user is mtcadmin.
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @author Nithika Senaweera <nithika.senaweera@mtc.co.uk>
     * @version 2.0
     * @copyright MTC 2025. <https://mtcmedia.co.uk>
     * */

    public function isMtcAdmin(Request $request): \Illuminate\Http\JsonResponse
    {

        // Check if the user is mtcadmin
        $is_mtcadmin = $this->admin_user->user['username'] === 'mtcadmin';

        return response()->json([
            'status' => 'success',
            'data' => $is_mtcadmin,
        ], 200);
    }

}
?>
