<?php

namespace Mtc\Plugins\GridManager\Http\Controllers;


use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Mtc\Plugins\GridManager\Models\GlobalRows;

class GridGlobalRowsController extends Controller
{
    protected $admin_user;

    /**
     * @description GlobalRowsController constructor.
     * */
    public function __construct()
    {
        $this->admin_user = new \AdminUser();
    }
    /**
     * @description Display a listing of the resource.
     */
    public function index()
    {
        return response()->json([
            'status' => 'success',
            'message' => 'Global rows loaded successfully',
        ], 200);
    }

    /**
     * @description Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Validate the request data
        $request->validate([
            'json' => 'required|json',
        ]);

        $jsonData = $request->input('json');

        // if json data exists in database, return error
        if (GlobalRows::where('json', $jsonData)->exists()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Global row with this JSON already exists',
            ], 400);
        }
        else {
            // Create a new GlobalRows instance
            $globalRow = new GlobalRows();
            $globalRow->json = $jsonData;
            $globalRow->last_modified_by = $this->admin_user->user['username'];
            $globalRow->ip_address = $request->ip();

            // Save the GlobalRows instance
            if ($globalRow->save()) {
                return response()->json([
                    'status' => 'success',
                    'message' => 'Global row created successfully',
                    'data' => $globalRow,
                ], 201);
            } else {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Failed to create global row',
                ], 500);
            }

        }

    }



    /**
     * Display the specified resource.
     */
    public function all(GlobalRows $globalRows)
    {
        $allrows = $globalRows::all();
        // get the json field from each row
        $rowsData = $allrows->map(function ($rows) {
            $rowJson = json_decode($rows->json, true);
            $rowJson['id'] = $rows->id;
            return $rowJson;
        });
        return response()->json([
            'status' => 'success',
            'data' => $rowsData,
        ], 200);
    }
    /**
     * @description Get the specified row by ID.
     * */
    public function getById(Request $request) {
        // Validate the request data
        $request->validate([
            'id' => 'required|integer|exists:grid_global_rows,id',
        ]);

        // Get the ID from the request
        $id = $request->input('id');
        // Find the GlobalRows instance by ID
        $globalRow = GlobalRows::find($id);
        if (!$globalRow) {
            return response()->json([
                'status' => 'error',
                'message' => 'Global row not found',
            ], 404);
        }
        // Return the GlobalRows instance
        return response()->json([
            'status' => 'success',
            'data' => json_decode($globalRow->json, true),
        ], 200);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, GlobalRows $globalRows)
    {
        // Validate the request data
        $request->validate([
            'json' => 'required|json',
            'id' => 'required|integer|exists:grid_global_rows,id',
        ]);
        // Get the ID from the request
        $id = $request->input('id');
        // Find the GlobalRows instance by ID
        $globalRow = GlobalRows::find($id);
        if (!$globalRow) {
            return response()->json([
                'status' => 'error',
                'message' => 'Global row not found',
            ], 404);
        }
        // Update the GlobalRows instance
        $globalRow->json = $request->input('json');
        $globalRow->last_modified_by = $this->admin_user->user;
        $globalRow->ip_address = $request->ip();
        // Save the changes
        if ($globalRow->save()) {
            return response()->json([
                'status' => 'success',
                'message' => 'Global row updated successfully',
                'data' => $globalRow,
            ], 200);
        } else {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to update global row',
            ], 500);
        }

    }

    /**
     * Remove the specified resource from storage.
     */
    public function delete(GlobalRows $globalRows, Request $request)
    {
        // Validate the request data
        $request->validate([
            'id' => 'required|integer|exists:grid_global_rows,id',
        ]);

        // Get the ID from the request
        $id = $request->input('id');
        // Find the GlobalRows instance by ID
        $globalRow = GlobalRows::find($id);
        if (!$globalRow) {
            return response()->json([
                'status' => 'error',
                'message' => 'Global row not found',
            ], 404);
        }
        // Delete the GlobalRows instance
        if ($globalRow->delete()) {
            return response()->json([
                'success' => true,
                'status' => 'success',
                'message' => 'Global row deleted successfully',
            ], 200);
        } else {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to delete global row',
            ], 500);
        }
    }
}
