<?php

namespace Mtc\Plugins\GridManager\Http\Controllers;
use Illuminate\Support\Facades\Cache;
use Mtc\Plugins\GridManager\Models\Grid as GridModel;
use Mtc\Plugins\GridManager\Models\GridStructures;

/**
 * Core Grid Manager class.
 *
 * @contributor     Jindich Prokop <dev.jp@mtcmedia.co.uk>
 * @copyright 2025 mtc. http://www.mtcmedia.co.uk/
 *
 * @version 0.2
 */
class GridManager
{

    /**
     * Path to templates, must be changed if plugin is not in its default folder
     *
     * @static string $template_path
     */
    public static string $template_path = 'plugins/GridManager/templates/';

    /**
     * @var string[]
     */
    public static array $allowed_grid_types = [
        'cms',
        'category',
        'product',
    ];

    /**
     * @param $id
     *
     * @return string
     */
    public static function getGridCacheKey($id): string
    {
        return "grid_{$id}";
    }

    /**
     * @param $id
     */
    public static function clearGridCache($id): void
    {
        $key = self::getGridCacheKey($id);

        if (Cache::has($key)) {
            Cache::forget($key);
        }
    }

    /**
     * Renders grid within CMS admin area
     *
     * @param array $list pagelist tying grid to page
     */
    public static function adminRenderGrid(array $list, int $id = null): void
    {
        $gridId = $id ?? ($list['type'] === 'grid' ? (int) $list[PAGE_LIST_COLUMN_COLUMN] : null);

        if (!$gridId) {
            return; // Or handle error/log
        }

        $grid = GridModel::find($gridId);

        if (!$grid) {
            return;
        }

        // Set required variables for the template
        $show_cms_actions = $id === null; // Only true if loaded from list
        $list['name']     = $grid->title;

        // Load template
        $templatePath = SITE_PATH . '/' . self::$template_path . 'admin/grid.tpl.php';

        if (file_exists($templatePath)) {
            require $templatePath;
        } else {
            error_log("Grid template not found at: $templatePath");
        }
    }


    /**
     * @description Renders grid structure in the frontend. used as a Twig function in the loader.
     * @param int $id The ID of the grid to render.
     * @param $twig Twig Environment to render the template with.
     * @return twig renders the grid structure.
     * @author Nithika Senaweera <nithika.senaweera@mtc.co.uk>
     * @version 2.0
     * @copyright MTC Media Ltd. 2025
     */
    public static function renderGrid(int $id, $twig) : void
    {
        // get the grid structure from the database
        $gridStructures = GridStructures::where('grid_id', $id)->first();

        if ($gridStructures) {
            // decode the JSON structure
            $json = json_decode($gridStructures->json, true);
        }
        else {
            // if the grid structure is not found, return an empty array
            $json = [];
        }


        // twig render the grid structure
        echo $twig->render('frontend/grid.twig', [
            'grid' => $json,
        ]);
    }


    /**
     * Adds grid information to pagedata.
     *
     * @param array $params Array of the form [
     *                      0 => PageList,
     *                      1 => array constructed pagelist representation,
     *                      2 => bool flag pagedata is being copied,
     *                      3 => bool flag if content is carried along,
     *                      [4 => Pagedata]
     *                      ]
     */
    public static function addToPagedataArray($params)
    {
        $page_list            = $params[0];
        $result               = &$params[1];
        $detached_for_copying = $params[2];
        $with_content         = $params[3];




        //Tiles not to be carried over when copying without content.
        if ($page_list->type == 'tile' && !$with_content) {
            $result = false;
        }

        //Only treat grids. Hook's firing for any page_list.
        if ($page_list->type != 'grid' || empty($grid_id = $page_list->{PAGE_LIST_COLUMN_COLUMN})) {
            return;
        }

        //Just remove pointer to grid when copying without content.
        if (!$with_content) {
            $result[PAGE_LIST_COLUMN_COLUMN] = '';

            return;
        }

        //Add grid image to logic containing only data needed by ::createGridCopyFromLogic()
        $grid = GridModel::find($grid_id);

        $result['grid'] = $grid;

        //Only the grid is being copied. Tiles need to be added.
        if (isset($params[4])) {
            $pagedata_ref              = &$params[4];
            $pagedata_ref->pagedata[0] = [];
            $tmp_pagedata              = new \Pagedata();

            $tmp_pagedata->get($pagedata_ref->page_id, 0, 0, 0, $with_content, $detached_for_copying, $grid->tiles->keys()
                                                                                                                  ->all());

            $pagedata_ref->pagedata = array_merge($pagedata_ref->pagedata, $tmp_pagedata->pagedata);
        }
    }

    /**
     * Changes page id of the grid version when page version is published
     *
     * @param $page_ids array of the form ['old_page_id' => $old_page_id, 'new_page_id' => $new_page_id]
     */
    public static function replacePageAndRelativePageId($page_ids)
    {
        GridModel::where('page_id', $page_ids['old_page_id'])
                 ->update(['page_id' => $page_ids['new_page_id']])
        ;
    }

    /**
     * Creates grids while pagedata is being pasted (for new page version being
     * created, page/pagedata being copypasted etc.)
     *
     * @param array $params Array of the form [
     *                      0 => Pagedata,
     *                      1 => int page id
     *                      ]
     */
    public static function reconstructGrids($params)
    {
        $pagedata = &$params[0];
        $page_id  = $params[1];

        $tiles_fix_ids = [];
        foreach ($pagedata->pagedata as $key => $page_datum) {
            if (isset($page_datum['type']) && $page_datum['type'] == 'grid') {
                //Prevent creating redundant copies
                if (GridModel::where('page_id', $page_id)
                             ->where('title', $page_datum['name'])
                             ->first()) {
                    continue;
                }

                if (!empty($page_datum['grid'])) {
                    //Content was passed ==> create new grid version with original content.
                    [$this_fix_ids, $new_grid] = self::createGridCopyFromLogic($page_datum['grid'], $page_id);
                    $tiles_fix_ids = $tiles_fix_ids + $this_fix_ids;
                } else {
                    //No content was passed ==> create empty grid.
                    $new_grid = GridModel::create([
                            'page_id' => $page_id,
                            'title'   => $page_datum['name'],
                        ]);
                }

                // Set the grid pointer to point to the new grid.
                $pagedata->pagedata[$key][PAGE_LIST_COLUMN_COLUMN] = $new_grid->id;
            }
        }

        //Fix tile --> column pointers.
        foreach ($pagedata->pagedata as $key => $page_datum) {
            if (isset($page_datum['type']) && $page_datum['type'] == 'tile') {
                if (!empty($tiles_fix_ids[$page_datum[PAGE_LIST_COLUMN_COLUMN]])) {
                    $pagedata->pagedata[$key][PAGE_LIST_COLUMN_COLUMN] = $tiles_fix_ids[$page_datum[PAGE_LIST_COLUMN_COLUMN]];
                }
            }
        }
        unset($tiles_fix_ids);
    }

    /**
     * Creates full copy of grid from supplied array with structure.
     * Returns an array of old => new page_list/tile ids to be fixed afterwards and newly created grid model.
     *
     * @param $original_grid
     * @param $new_page_id
     *
     * @return array
     */
    public static function createGridCopyFromLogic($original_grid, $new_page_id)
    {
        $fix_ids = [];

        $new_grid = GridModel::create([
            'page_id' => $new_page_id,
            'title'   => $original_grid['title'],
        ]);

        return [$fix_ids, $new_grid];
    }


    /**
     * @param string $type
     * @param int $page_id
     *
     * @return array
     */
    public static function getGridByType(string $type, int $page_id): array
    {
        if (empty($page_id) || empty($type)) {
            return [];
        }

        $record = GridModel::where('type', $type)
                      ->where('page_id', $page_id)
                      ->first();

        return $record ? $record->toArray() : [];
    }


}
