<?php
/**
 * Coupon Conditon Eloquent model
 *
 * @version 22/09/16
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Shop\Coupon;

use Mtc\Shop\Coupon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

/**
 * Coupon Condition Eloquent model.
 * Implements Shop coupon main functionality
 *
 * @version 22/09/16
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class Condition extends Model
{

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'coupons_conditions';

    /**
     * @var array The attributes that are mass assignable.
     */
    protected $fillable = [
        'coupon_id',
        'type',
        'exclude',
        'value'
    ];

    /**
     * Define condition types
     * @var array
     */
    public static $types = [
        'item',
        'category',
        'exclude_category',
        'brand',
        'exclude_brand'
    ];

    /**
     * Scope - ofCoupon()
     *
     * @param Builder $query Query to extend
     * @param mixed $coupon reference of coupon (allows int id, array or object)
     * @return Builder
     */
    public function scopeOfCoupon(Builder $query, $coupon)
    {
        return $query->where("coupon_id", id($coupon));
    }

    /**
     * Scope - excluded()
     *
     * @param Builder $query Query to extend
     * @param int $excluded whether condition is exclusive or inclusive
     * @return Builder
     */
    public function scopeExcluded(Builder $query, $excluded = 1)
    {
        return $query->where("exclude", $excluded);
    }

    /**
     * Define Eloquent relationship to coupon class
     * @return BelongsTo
     */
    public function coupon()
    {
        return $this->belongsTo(Coupon::class, 'coupon_id', 'id');
    }

    /**
     * Update conditions for a coupon type.
     * @param Coupon $coupon
     * @param $type
     * @param array $request
     */
    public static function updateByType(Coupon $coupon, $type, $request = [])
    {
        // filter out empty entries
        $request = array_filter($request);

        $exclude = strpos($type, 'exclude') === 0 ? 1 : 0;
        $type = str_replace('exclude_', '', $type);

        //Remove conditions that have been removed
        self::ofCoupon($coupon->id)
            ->where('type', $type)
            ->where('exclude', $exclude)
            ->whereNotIn('value', $request)
            ->delete();

        foreach ($request as $entry) {
            $exists = self::ofCoupon($coupon->id)
                ->where('type', $type)
                ->where('exclude', $exclude)
                ->where('value', $entry)
                ->count() > 0;
            if (!$exists) {
                self::create([
                    'coupon_id' => $coupon->id,
                    'type' => $type,
                    'exclude' => $exclude,
                    'value' => $entry
                ]);
            }
        }

    }
}