<?php

use Mtc\Core\Currency;
use Illuminate\Support\Collection;
use Mtc\Core\Permissions;
use Mtc\Modules\Members\Classes\Auth;
use Twig\Extra\Intl\IntlExtension;
use Twig\Extra\String\StringExtension;
use Twig\TwigFilter;
use Twig\TwigFunction;
use Mtc\Plugins\MembersMessaging\Models\Thread;

/*
 *   Extends Twig with a function that allows you to check if something is
 * numeric
 */


$test = new \Twig\TwigTest('numeric', function ($value) {
    return is_numeric($value);
});

$twig->addTest($test);

/*
 *   Add twig function for returning execution time of a script.
 */

$twig->addFunction(new TwigFunction('execution_time', function () {
    if (DEV_MODE) {
        return round(microtime(true) - START_TIME, 4) . 's';
    }
}));

/*
 *
 *  Expose the date_create_from_format function to Twig
 *  use as {{ create_from_format($date, 'd/m/Y H:i', 'jS F Y') }}
 *
 */

$twig->addFunction(new TwigFunction('create_from_format', function ($date, $format_from, $format_to) {
    if (!empty($date) && !empty($format_from) && !empty($format_to)) {
        return date_create_from_format($format_from, $date)->format($format_to);
    }
}));

/*
 *  Generate a hash value using the contents of a given file
 */

$twig->addFunction(new TwigFunction('uriHash', function ($uri) {
    return $uri . '?_' . hash_file('crc32', SITE_PATH . $uri);
}));

$twig->addFunction(new TwigFunction('md5', function ($string) {
    return md5($string);
}));

$twig->addFunction(new TwigFunction('copyRightRange', function ($date) {
    $ret = date('Y', strtotime($date));
    if ($ret < date('Y')) {
        $ret = $ret . ' &ndash; ' . date('Y');
    }
    return $ret;
}));

/*
 *  Print_r for twig
 */

$twig->addFunction(new TwigFunction('e', function ($array) {
    echo '<pre>';
    print_r($array);
    echo '</pre>';
}));

/*
 *
 *  Hooks function
 *  {{ action('header') }} would introduce a new action filter inside a template
 * file
 *
 */

$twig->addFunction(new TwigFunction('hooks_do_action', function () {
    call_user_func_array('HooksAdapter::do_action', func_get_args());
}));

$twig->addFunction(new TwigFunction('site_menu', function($menu_name) {
    static $cache = [];
    if (array_key_exists($menu_name, $cache)) {
        return $cache[$menu_name];
    }

    $cache[$menu_name] = \Mtc\Plugins\SiteMenu\Classes\SiteMenu::fetchTree($menu_name);
    return $cache[$menu_name];
}));

/*
 *
 *  CSRF function
 *  {{ csrf_token('form_name') }} would add a security token to form
 *
*/
$twig->addFunction(new TwigFunction('csrf_token', function ($form_name) {
    return CSRF::token($form_name);
}));

$twig->addFunction(new TwigFunction('load_assets', function () {
    (new AssetLoader)->all();
}));

/*
 * Check if an array contains an array or object that has an attribute
 * with given value.
 * @param $needle value of attribute
 * @param @array_of_objects
 * @param @attribute_name
 * @return bool $success
 */
$twig->addFunction(
    new TwigFunction('in_array_of_objects_or_array',
        function ($needle, $array_of_objects, $attribute_name = "id") {
            if (empty($array_of_objects)) {
                return false;
            }
            foreach ($array_of_objects as $object) {
                if (is_array($object)) {
                    $array = $object;
                } else {
                    $array = get_object_vars($object);
                }
                if (isset($array[$attribute_name]) && $array[$attribute_name] == $needle) {
                    return true;
                }
            }
            return false;
        }));

/**
 * get contents of file
 * @param $file string
 * @return $file string
 * @author Kieran McBurney <kieran.mcburney@mtcmedia.co.uk>
 */
$twig->addFunction(
    new TwigFunction('file_get_contents', function ($file) {

        $fileinfo = pathinfo($file);
        $allowed_extensions = array('svg');

        if (in_array(strtolower($fileinfo['extension']), $allowed_extensions) && file_exists($file)) {
            $file = file_get_contents($file);
            return $file;
        } else {
            return '';
        }
    }
    ));

/**
 * get the current URI
 * @return string
 * @author Aidan Ferguson <aidan.ferguson@mtcmedia.co.uk>
 */
$twig->addFunction(new TwigFunction('current_uri', function ($stip_vars = false) {

    $uri = $_SERVER['REQUEST_URI'];

    if ($stip_vars) {
        $uri = preg_replace('/\?.*?$/', '', $uri);
    }


    return $uri;
}));


/**
 * Filter user input of telephone number for output
 * @return string
 * @author Andrew Morgan <andrew.morgan@mtcmedia.co.uk>
 */
$twig->addFunction(new TwigFunction('filter_tel', function ($unsanitized_phone_number) {

    // Filter out all characters except numbers 0 to 9 and the plus character
    $unsanitized_phone_number = preg_replace('/[^0-9+]/', '', $unsanitized_phone_number);

    //matches country codes like +44 or 0044 at the start of a string when followed by a 0
    $regex = '/(^(\+|00)[1-9]{1,3})0/';

    //strip the predial 0 if a country code is used
    $sanitized_phone_number = preg_replace($regex, '$1', $unsanitized_phone_number);

    return $sanitized_phone_number;

}));

/**
 * casts values as strings and compares them if they are equal
 * @return bool
 * @author Rihards Silins <rihards.silins@mtcmedia.co.uk>
 */
$twig->addFunction(new TwigFunction('strings_are_equal', function ($first_value, $second_value) {
    return ((string)$first_value === (string)$second_value);
}));

/**
 * currency_format filter
 *
 * @param float $amount number to output
 * @return string the number in correct currency format
 */
$twig->addFilter(new TwigFilter('currency_format', function ($amount) {

    $sign = '';

    if ($amount < 0) {
        $sign = '-';
        $amount = -$amount;
    }

    $currency = $_SESSION['currency'] ?? Currency::getCurrencyFromCode();

    return $sign . $currency['symbol']
        . number_format($amount * $currency['ratio'], 2);
}, [
    'is_safe' => ['html']
]));

$twig->addFilter(new TwigFilter('truncate', function ($value, $length = 30, $preserve = false, $separator = '...') {
    if (mb_strlen($value) <= $length) {
        return $value;
    }

    if ($preserve) {
        $breakpoint = mb_strpos($value, ' ', $length);
        if ($breakpoint !== false) {
            $length = $breakpoint;
        }
    }

    return mb_substr($value, 0, $length) . $separator;
}));

/**
 * Return enabled currencies
 *
 * @return Array Returns currency object of all enabled currencies
 * @author Andrew Morgan <andrew.morgan@mtcmedia.co.uk>
 *
 */
$twig->addFunction(new TwigFunction('enabled_currencies', function () {
    return Currency::getEnabledCurrencies();
}));

/**
 * Ability to call a static method through twig
 *
 * @param string $class
 * @param string $function
 * @param array $args
 *
 * @return mixed|null
 * @author Lukas Giegerich <lukas@mtcmedia.co.uk>
 * @author Stackoverflow ({@link http://stackoverflow.com/questions/6844266/twig-template-engine-using-a-static-function-or-variable})
 *
 */


$twig->addFunction(new TwigFunction('staticCall', function ($class, $function, $args = []) {
    if (class_exists($class) && method_exists($class, $function)) {
        return call_user_func_array([$class, $function], $args);
    }

    return null;
}));

/**
 * Add a new Twig function for rendering form fields
 */
$twig->addFunction(new TwigFunction('form_field', function ($options) use ($twig) {
    return \Mtc\Core\FormField::renderField($options, $twig);
}, [
    'is_safe' => ['html']
]));

/*
 * Creates the html for all the available currencies.
 */

$filter = new TwigFilter('multi_currency', function ($amount) use ($twig) {

    $currency_data = Currency::getCurrencyFromCode($_SESSION['selected_currency'] ?? null);
    $options = [
        'amount' => (float)number_format($amount, 2),
        'session_currency' => $currency_data,
    ];

    return $twig->render('currency/multi_currency.twig', $options);

}, [
    'is_safe' => ['html']
]);

$twig->addFilter($filter);

$twig->addFilter(new TwigFilter('sanitized_html', function ($value) {
    return clean_page($value, true, true);
}, [
    'is_safe' => ['html']
]));

$twig->addFunction(new TwigFunction('mergeMenus', function ($items, $categories) {
    $menu = new Collection();
    $menu = $menu->merge($items);
    $menu = $menu->merge($categories);
    return $menu->slice(0, 8)->all();
}));


$twig->addFunction(new TwigFunction('method_field', function ($method) {
    return method_field($method);
}, [
    'is_safe' => ['html']
]));

/**
 * Generate the URL to a named route.
 *
 * @param array|string $name
 * @param mixed $parameters
 * @param bool $absolute
 * @return string
 */
$twig->addFunction(new TwigFunction('route', function ($name, $parameters = [], $absolute = true) {
    return route($name, $parameters, $absolute);
}));

$twig->addFunction(new TwigFunction('routeName', function () {
    return request()->route()->getName();
}));

$twig->addFunction(new TwigFunction('event', function ($event, $parameters) {
    $templates = \Illuminate\Support\Facades\Event::dispatch($event, new $event($parameters));
    foreach ($templates as $markup) {
        echo $markup;
    }
}));

$twig->addFunction(new TwigFunction('can', function ($permission) {
    return Permissions::can($permission);
}));


$twig->addFunction(new TwigFunction('file_exists', function ($url) {
    if (file_exists($_SERVER["DOCUMENT_ROOT"] . $url)) {
        return true;
    }
}));

/**
 * Load values from previous request
 *
 * @param string $key
 * @param mixed $default
 * @return string
 */
$twig->addFunction(new TwigFunction('old', function ($key, $default = null) {
    return old($key, $default);
}));

$twig->addFunction(new TwigFunction('config', function ($key = null, $default = null) {
    return config($key, $default);
}));
$twig->addFunction(new TwigFunction('request', function ($key = null, $default = null) {
    return request($key, $default);
}));

/**
 * Find the number of member new messages
 */
$twig->addFunction(new TwigFunction('member_new_messages', function () {
    $member = Auth::getLoggedInMember();

    if (!$member) {
        return '';
    }

    $threads = Thread::query()
        ->whereNull('deleted_at')
        ->where('member_id', $member->id)
        ->get();

    $unreadCount = $threads->sum(function ($thread) use ($member) {
        return $thread->getMemberUnreadMessagesCount($member->id);
    });

    return $unreadCount > 0 ? "( {$unreadCount} )" : '';
}));

$twig->addFunction(new TwigFunction('is_active_url', function($pattern) {
    return request()->routeIs($pattern);
}));
