<?php

namespace Mtc\Plugins\MembersMessaging\Http\Controllers;

use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Mtc\Core\Admin\Role;
use Mtc\Core\Permissions;
use Mtc\Plugins\MembersMessaging\Models\Template;

/**
 * Controller for Templates
 */
class TemplatesController extends Controller
{
    /**
     * @return Template
     */
    public function index()
    {
        $templates = Template::query()
            ->orderBy('id', 'ASC')
            ->paginate(25)
            ->appends(request()->query());

        $totalnumrows = $templates->total();

        return app('twig')->render('membersmessaging/admin/templates.twig', [
            'can_edit_templates' => Permissions::can(Permissions::EDIT_MESSAGE_TEMPLATES),
            'current_path' => request()->url(),
            'templates' => $templates,
            'totalnumrows' => $totalnumrows,
            'success' => session('success'),
            'error' => session('error'),
            'warning' => session('warning'),
        ]);
    }

    /**
     * @return Template
     */
    public function create()
    {
        return app('twig')->render('membersmessaging/admin/templates_create.twig', [
            'can_edit_templates' => Permissions::can(Permissions::EDIT_MESSAGE_TEMPLATES),
            'current_path' => request()->url(),
            'roles' => Role::all(),
            'categories' => Template::$categories,
        ]);
    }

    /**
     * @param Request $request
     * @return RedirectResponse
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'category' => 'required|string',
            'role' => 'required|integer',
            'name' => 'required|string',
            'text' => 'required|string',
        ]);

        $template = new Template();
        $template->fill($validated);

        $template->save();

        return redirect()->route('admin.messaging.templates')->with('success', 'Template added successfully.');
    }

    /**
     * @return void
     */
    public function show() {}

    /**
     * @param Request $request
     * @param $id
     * @return Template
     */
    public function edit(Request $request, $id)
    {

        $template = Template::find($id);

        return app('twig')->render('membersmessaging/admin/templates_edit.twig', [
            'can_edit_templates' => Permissions::can(Permissions::EDIT_MESSAGE_TEMPLATES),
            'current_path' => request()->url(),
            'template' => $template,
            'roles' => Role::all(),
            'categories' => Template::$categories,
        ]);
    }

    /**
     * @param Request $request
     * @param $id
     * @return RedirectResponse
     */
    public function update(Request $request, $id)
    {

        $validated = $request->validate([
            'category' => 'required|string',
            'role' => 'required|integer',
            'name' => 'required|string',
            'text' => 'required|string',
        ]);

        $template = Template::find($id);
        $template->fill($validated);
        $template->text = $validated['text'];

        $template->save();

        return redirect()->route('admin.messaging.templates', $id)->with('success', 'Template updated successfully.');
    }

    /**
     * @param Request $request
     * @param $id
     * @return RedirectResponse
     */
    public function destroy(Request $request, $id)
    {
        Template::find($id)->delete();

        return back()->with('success', 'Template deleted.');
    }

    public function fetchTemplate($id)
    {
        $template = Template::find($id);

        if (!$template) {
            return response()->json([
                'success' => false,
                'message' => 'Template not found.'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'text' => $template->text
        ]);
    }
}
