<?php

namespace App\Http\Controllers;

use App\Services\ComponentScannerService;
use App\Services\ComponentRendererService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class ComponentRegistryApiController extends Controller
{
    private ComponentScannerService $scannerService;
    private ComponentRendererService $rendererService;

    public function __construct(
        ComponentScannerService $scannerService,
        ComponentRendererService $rendererService
    ) {
        $this->scannerService = $scannerService;
        $this->rendererService = $rendererService;
    }

    /**
     * GET /api/components
     * Return list of all discoverable components with metadata
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $categoryFilter = $request->query('category');
            $searchQuery = $request->query('search');
            $limit = min((int) $request->query('limit', 50), 200);
            $page = max((int) $request->query('page', 1), 1);

            // Get all components
            $discovery = $this->scannerService->discoverComponents();
            $allComponents = $discovery['components'];
            $categories = $discovery['categories'];

            // Apply filters
            $filteredComponents = $allComponents;

            if ($categoryFilter) {
                $filteredComponents = array_filter($filteredComponents, function($component) use ($categoryFilter) {
                    return $component->category === $categoryFilter;
                });
            }

            if ($searchQuery) {
                $filteredComponents = array_filter($filteredComponents, function($component) use ($searchQuery) {
                    $query = strtolower($searchQuery);
                    return str_contains(strtolower($component->name), $query) ||
                           str_contains(strtolower($component->description), $query);
                });
            }

            // Convert to indexed array for pagination
            $filteredComponents = array_values($filteredComponents);
            $total = count($filteredComponents);

            // Apply pagination
            $offset = ($page - 1) * $limit;
            $paginatedComponents = array_slice($filteredComponents, $offset, $limit);

            // Convert to array format
            $componentsData = array_map(function($component) {
                return $component->toArray();
            }, $paginatedComponents);

            return response()->json([
                'data' => $componentsData,
                'meta' => [
                    'total' => $total,
                    'per_page' => $limit,
                    'current_page' => $page,
                    'last_page' => ceil($total / $limit),
                ],
                'categories' => $categories,
            ]);

        } catch (\Exception $e) {
            \Log::error('Component Registry API Error: ' . $e->getMessage());

            return response()->json([
                'error' => 'Failed to scan component directory',
                'details' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * GET /api/components/{path}
     * Return detailed information about a specific component
     */
    public function show(Request $request, string $path): JsonResponse
    {
        try {
            $component = $this->scannerService->findComponent($path);

            if (!$component) {
                return response()->json([
                    'error' => 'Component not found',
                    'path' => $path
                ], 404);
            }

            // Get component with sample data and usage instructions
            $componentData = $this->rendererService->getComponentWithSampleData($component);

            return response()->json($componentData);

        } catch (\Exception $e) {
            \Log::error("Component Detail API Error for {$path}: " . $e->getMessage());

            return response()->json([
                'error' => 'Failed to load component details',
                'details' => config('app.debug') ? $e->getMessage() : 'Internal server error',
                'path' => $path
            ], 500);
        }
    }
}