# 🧪 Internal Git Workflow – Pharmacy Project

> For internal development team members working on the Pharmacy platform.

---

## 📦 Cloning the Repository

```bash
git clone git@bitbucket.org:mtcmedia/pharmacy-system-starter.git ~/pharmacy-starter
cd ~/pharmacy-starter
```

> ℹ️ After cloning, set up the project locally using Docker or on the dev server. Make sure dependencies are installed and the project is runnable before continuing.

---

## 🌿 Git Branching Strategy

We follow a **semantic branching model** for clarity and consistency:

| Branch Type     | Purpose                    | Example                     |
|-----------------|----------------------------|-----------------------------|
| `feature/*`     | New feature development    | `feature/user-login-ui`     |
| `bugfix/*`      | Bug or issue fix           | `bugfix/order-total-rounding` |
| `hotfix/*`      | Critical fix               | `hotfix/crash-on-checkout`  |
| `improvement/*` | General Enhancement        | `improvement/refactor-auth`  |
| `release/*`     | First version of a release | `release/rc1.0`             |

> ✅ Always branch off `master` unless otherwise stated.

---

## 🛠 Common Git Commands

### 🧾 Listing Branches

```bash
# Local branches
git branch

# Remote branches
git branch -r
```

### 🌱 Creating & Switching Branches

```bash
# Create and switch to a new branch
git checkout -b feature/your-branch-name

# Switch to existing local branch
git checkout feature/your-branch-name
```

### 🔄 Merging Branches

```bash
# Merge another branch into current one
git merge branch-name
```

### ⬆️ Updating Your Branch from Master

```bash
git checkout master
git pull
git checkout feature/your-branch-name
git merge master
```

---

## 🚀 Git Workflow – Step by Step

### Step 1: Create a Feature Branch

```bash
git checkout master
git pull origin master
git checkout -b feature/your-feature-name
git push -u origin feature/your-feature-name
```

---

### Step 2: Make Your Changes

```bash
# Add all changes
git add .

# Or add selected files 
git add path/to/file

# Commit with a meaningful message
git commit -m "implement X feature"
```

---

### Step 3: Keep Your Branch in Sync

```bash
# Pull latest changes
git pull origin feature/your-feature-name

# Push your work
git push
```

---

### Step 4: Create a Pull Request

1. Test your feature.
2. Push final changes.
3. Go to **Bitbucket**, open a **Pull Request** from your feature branch into `master`.
4. Request code review and assign reviewers.
5. Reviewers will merge only **after review and tasks/tests pass**.

---

## 👥 Example Collaborative Workflow (Shared Feature Branch)

### Backend Dev

```bash
git checkout master
git pull origin master
git checkout -b feature/media-manager
git push -u origin feature/media-manager
```

### Frontend Dev

If repo not cloned yet:

```bash
git clone git@bitbucket.org:your-team/project.git
cd project
git fetch origin
git checkout -b feature/media-manager origin/feature/media-manager
```

If already cloned:

```bash
cd project
git fetch origin
git checkout -b feature/media-manager origin/feature/media-manager
```

> 🔄 Always pull before pushing:
```bash
git pull origin feature/media-manager
```

> 💾 After changes:
```bash
git add .
git commit -m "feat: your message"
git push origin feature/media-manager
```

---

## 🔒 Automatically Closing Issues in Bitbucket

When committing changes, you can use one of the following **closing keywords** to automatically close the referenced Bitbucket issue when the commit is merged into the default branch (`main`, `master`, etc.):

- `closes`
- `fixes`
- `resolves`

✅ **These keywords function the same way** in Bitbucket. The issue will be closed automatically upon merge.

### 💡 Example Usage:
```bash
git commit -m "Fixed XSS vulnerability. Resolves #4"
git commit -m "Patched SQL injection. Fixes #23"
git commit -m "Updated dependency due to security issue. Closes #78"
```

---

## 🤔 Are There Any Differences in the keywords?

No functional difference — just a matter of **semantic preference**.

| Keyword   | Best Used For                            |
|-----------|-------------------------------------------|
| `fixes`   | When fixing a **bug** or **security issue** |
| `closes`  | When completing a **task** or **feature**   |
| `resolves`| When resolving a **problem** or **enhancement** |

---

## 🔁 Keeping Your Local Branch Updated

### With Uncommitted Changes

```bash
git stash
git pull
git stash pop
```

### Without Uncommitted Changes

```bash
git pull
```

---

## 📌 Check Remote Tracking

### Option 1: Using `git status`

```bash
git status
# Output should mention "Your branch is up to date with ..."
```

### Option 2: Using `git branch -vv`

```bash
git branch -vv
# Shows local branches and their upstream remotes
```

### Option 3: Config Check

```bash
git config --get branch.feature/my-branch.remote
```

> 🛠 Set tracking if needed:
```bash
git branch --set-upstream-to=origin/feature/my-branch
```

---
📺 [Watch Git Workflow Tutorial Video](https://mtcmedia-my.sharepoint.com/:v:/g/personal/efraim_zakarias_mtc_co_uk/EfNXxABEpdJNrDrDRkHLmWAB-rFejSDH22zo7hCGLVQ9ZQ?nav=eyJyZWZlcnJhbEluZm8iOnsicmVmZXJyYWxBcHAiOiJPbmVEcml2ZUZvckJ1c2luZXNzIiwicmVmZXJyYWxBcHBQbGF0Zm9ybSI6IldlYiIsInJlZmVycmFsTW9kZSI6InZpZXciLCJyZWZlcnJhbFZpZXciOiJNeUZpbGVzTGlua0NvcHkifX0&e=Am7GNs)

---
## 📥 Summary Cheatsheet

```bash
# Clone project
git clone [repo-url]

# Start new feature
git checkout master
git pull origin master
git checkout -b feature/new-feature
git push -u origin feature/new-feature

# Make changes
git add .
git commit -m "added X feature. Closes #23"
git push

# Sync updates
git pull

# Create PR → Review
```
