const treatmentsSignature = function (list) {
    if (!Array.isArray(list)) {
        return '[]';
    }
    try {
        return JSON.stringify(list);
    } catch (e) {
        return String(list.length || 0);
    }
};


Vue.component('patientTreatmentPlan', {
    props: ['routes','events','initialTreatments'],

    data: function() {
        const initialProvided = Array.isArray(this.initialTreatments);
        const initial = initialProvided ? this.initialTreatments.slice() : [];
        return {
            treatments: initial,      // always an array
            hasLoaded: initialProvided,
            loading: false,
            add: false,
            showAll: false,
            eventIndex: 0,
            bootstrapTreatmentsSignature: treatmentsSignature(initial),
            treatmentsOverlayBound: false,
        };
    },

    mounted() {
        this.init();
    },

    computed: {
        canLoadAll() {
            const list = Array.isArray(this.treatments) ? this.treatments : [];
            return list.length > 1 && !this.showAll;
        },
        activeTreatment() {
            const list = Array.isArray(this.treatments) ? this.treatments : [];
            if (!list.length) return null;
            // tolerate nulls/partials
            for (let i = 0; i < list.length; i++) {
                const t = list[i];
                if (t && t.status === 'active') return t;
            }
            return null;
        },
    },

    watch: {
        initialTreatments: {
            immediate: true,
            handler(list) {
                const next = Array.isArray(list) ? list : [];
                const signature = treatmentsSignature(next);
                // No-op if identical to what we already applied from bootstrap
                if (signature === this.bootstrapTreatmentsSignature) {
                    return;
                }
                this.bootstrapTreatmentsSignature = signature;
                this.treatments = next.slice();
                this.hasLoaded = true;
                this.loading = false;
                // Fire once (or when changed) to allow parent to (re)bind any overlays
                this.notifyTreatmentsLoaded();
            }
        },
        events(stack) {
            for (let i = this.eventIndex; i < stack.length; i++) {
                const event = stack[i];
                if (event.name === 'treatmentSaved' || event.name === 'treatmentDeleted') {
                    this.load(true);
                }
            }
            this.eventIndex = stack.length;
        }
    },

    methods: {
        init() {
            if (this.hasLoaded) {
                this.notifyTreatmentsLoaded();
                return;
            }
            this.load(true);
        },

        editTreatmentAction(treatment) {
            // Ensure defaultTreatment exists somewhere in scope
            if (typeof treatment === 'undefined') {
                if (typeof defaultTreatment === 'undefined') {
                    // fallback to a minimal empty object to avoid runtime errors
                    treatment = { status: null };
                } else {
                    treatment = defaultTreatment;
                }
            }
            this.emitEvent('initEditTreatment', { treatment });
        },

        cancel() {
            this.add = false;
        },

        load(force = false) {
            const vm = this;
            if (!force && this.hasLoaded) {
                return;
            }

            const applyPayload = (payload) => {
                const list = Array.isArray(payload?.treatments) ? payload.treatments : [];
                const signature = treatmentsSignature(list);
                const changed = signature !== vm.bootstrapTreatmentsSignature;
                vm.treatments = list.slice();
                vm.hasLoaded = true;
                vm.loading = false;
                if (changed) {
                    vm.bootstrapTreatmentsSignature = signature;
                }
                // Only notify parent to (re)bind overlays when first bound or data actually changed
                vm.notifyTreatmentsLoaded(changed);
                // Cache only when content changed
                if (changed && typeof vm.cacheBootstrap === 'function') {
                    vm.cacheBootstrap('admin-patient-treatments-load', { treatments: list.slice() });
                }
            };

            const cached = (!force && this.consumeBootstrap)
                ? this.consumeBootstrap('admin-patient-treatments-load')
                : null;
            if (cached) {
                applyPayload(cached);
                return;
            }

            this.loading = true;

            this.$http.post(this.routes['admin-patient-treatments-load'], {})
                .then((response) => {
                    let payload = response.data;

                    if (typeof payload === 'string') {
                        try { payload = JSON.parse(payload); }
                        catch (e) {
                            console.error('Invalid JSON from admin-patient-treatments-load', e, payload);
                            payload = {};
                        }
                    }

                    applyPayload(payload);
                })
                .catch((err) => {
                    console.error('Treatments load failed:', err);
                    vm.loading = false;
                    vm.treatments = [];
                });
        },

        loadAll() {
            this.showAll = true;
        },

        emitEvent(name, data) {
            this.$parent.emitEvent(name, data);
        },
        notifyTreatmentsLoaded(force = false) {
            if (!this.treatmentsOverlayBound || force) {
                this.treatmentsOverlayBound = true;
                this.emitEvent('treatmentsLoaded');
            }
        },
    },
});
