<?php

namespace Mtc\Shop\Assessment;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

/**
 * Assessment Ethnicity Model
 *
 * Manages ethnicity categories for assessments
 * Provides a centralized way to manage different ethnic groups
 *
 * @property int $id
 * @property string $name
 * @property string|null $description
 * @property bool $is_active
 * @property \Carbon\Carbon $created_at
 * @property \Carbon\Carbon $updated_at
 */
class AssessmentEthnicity extends Model
{
    protected $table = 'assessment_ethnicities';

    protected $fillable = [
        'name',
        'description',
        'sort_order',
        'is_active',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'sort_order' => 'integer',
    ];

    /**
     * Get the BMI ethnicity records for this ethnicity
     *
     * @return HasMany
     */
    public function bmiEthnicities(): HasMany
    {
        return $this->hasMany(BmiEthnicity::class, 'ethnicity_id');
    }

    /**
     * Scope to get only active ethnicity records
     *
     * @param Builder $query
     * @return Builder
     */
    public function scopeActive(Builder $query): Builder
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope to filter by name
     *
     * @param Builder $query
     * @param string $name
     * @return Builder
     */
    public function scopeByName(Builder $query, string $name): Builder
    {
        return $query->where('name', 'LIKE', "%{$name}%");
    }

    /**
     * Get all active ethnicities
     *
     * @return \Illuminate\Support\Collection
     */
    public static function getActiveEthnicities(): \Illuminate\Support\Collection
    {
        return self::active()->orderBy('sort_order')->orderBy('name')->get();
    }

    /**
     * Find ethnicity by name
     *
     * @param string $name
     * @return self|null
     */
    public static function findByName(string $name): ?self
    {
        return self::active()->where('name', $name)->first();
    }

    /**
     * Create default ethnicity records
     *
     * @return void
     */
    public static function createDefaultEthnicities(): void
    {
        $defaultEthnicities = [
            'Asian or Asian British' => 1,
            'Black, African, Caribbean or Black British' => 2,
            'Middle Eastern' => 3,
            'Mixed or multiple ethnicities with an Asian, Black or Middle Eastern background' => 4,
            'White' => 5,
            'Other ethnic group' => 6,
        ];

        foreach ($defaultEthnicities as $ethnicityName => $sortOrder) {
            if (!self::where('name', $ethnicityName)->exists()) {
                self::create([
                    'name' => $ethnicityName,
                    'description' => "Default ethnicity category: {$ethnicityName}",
                    'sort_order' => $sortOrder,
                    'is_active' => true,
                ]);
            }
        }
    }

    /**
     * Get ethnicity with BMI categories
     *
     * @return array
     */
    public function getWithBmiCategories(): array
    {
        return [
            'id' => $this->id,
            'name' => $this->name,
            'description' => $this->description,
            'bmi_categories' => $this->bmiEthnicities()
                ->active()
                ->get()
                ->map(function ($bmiEthnicity) {
                    return [
                        'category' => $bmiEthnicity->bmi_category,
                        'cutoff_low' => $bmiEthnicity->cutoff_low,
                        'cutoff_high' => $bmiEthnicity->cutoff_high,
                        'message' => $bmiEthnicity->message,
                    ];
                })
                ->toArray(),
        ];
    }
}
