Vue.component('addPatientNote', {
    props: ['routes', 'addNote', 'params'],

    data() {
        return {
            form: { id: null, content: '' },
            loading: false,
            errors: {},
            _tinymce: null,
            _tinymceBound: false,
            _attachTimer: null
        };
    },

    mounted() {
        if (this.addNote) this.applyParams(this.params);
        this.bindWysiwyg();
    },

    beforeDestroy() {
        if (this._attachTimer) clearTimeout(this._attachTimer);
        if (this._tinymce && this._tinymceBound) {
            this._tinymce.off('keyup change input');
            this._tinymceBound = false;
        }
    },

    watch: {
        addNote(val) {
            if (val) {
                this.applyParams(this.params);
                this.$nextTick(() => {
                    this.bindWysiwyg();
                    this.seedEditorFromForm();
                });
            } else {
                this.resetForm();
                this.clearEditor();
            }
        },
        params: {
            deep: true,
            handler(p) {
                if (this.addNote) {
                    this.applyParams(p);
                    this.$nextTick(() => this.seedEditorFromForm());
                }
            }
        },
        'form.content'(val) {
            if ((val || '').trim().length && this.errors.general) {
                this.$delete(this.errors, 'general');
            }
        }
    },

    methods: {
        bindWysiwyg() {
            const tryAttach = () => {
                if (window.tinymce) {
                    const ed = tinymce.get('patientNotesInput');
                    if (ed) {
                        this._tinymce = ed;

                        if (!this._tinymceBound) {
                            const syncFromEditor = () => {
                                this.form.content = this._tinymce.getContent({ format: 'html' }) || '';
                                const text = (this._tinymce.getContent({ format: 'text' }) || '').trim();
                                if (text && this.errors.general) this.$delete(this.errors, 'general');
                            };
                            this._tinymce.on('keyup change input', syncFromEditor);
                            this._tinymceBound = true;
                        }

                        if (typeof ed.getDoc === 'function' && ed.getDoc()) {
                            this.seedEditorFromForm();
                        } else {
                            ed.once('init', () => this.seedEditorFromForm());
                        }
                        return;
                    }
                }
                this._attachTimer = setTimeout(tryAttach, 150);
            };
            tryAttach();
        },

        seedEditorFromForm() {
            if (!this._tinymce) return;
            const setNow = () => {
                if (typeof this._tinymce.getDoc === 'function' && this._tinymce.getDoc()) {
                    this._tinymce.setContent(this.form.content || '');
                    if (this._tinymce.undoManager) this._tinymce.undoManager.clear();
                } else {
                    this._tinymce.once('init', setNow);
                }
            };
            setNow();
        },

        clearEditor() {
            if (!this._tinymce) return;
            const clearNow = () => {
                if (typeof this._tinymce.getDoc === 'function' && this._tinymce.getDoc()) {
                    this._tinymce.setContent('');
                    if (this._tinymce.undoManager) this._tinymce.undoManager.clear();
                } else {
                    this._tinymce.once('init', clearNow);
                }
            };
            clearNow();
        },

        applyParams(p) {
            this.resetForm();
            if (p && p.id) {
                this.form.id = p.id;
                // accept either p.content (HTML) or p.note (plain text)
                this.form.content = p.content || p.note || '';
            }
        },

        resetForm() {
            this.errors = {};
            this.form = { id: null, content: '' };
        },

        close() {
            this.$parent.emitEvent('hideAddNote');
            $('.overlayCloseButton').trigger('click');
        },

        cancel() { this.close(); },

        store() {
            if (this.loading) return;
            this.errors = {};

            // Collect HTML + TEXT safely (TinyMCE or fallback)
            let html = (this.form.content || '').trim();
            let text = '';

            if (this._tinymce) {
                html = (this._tinymce.getContent({ format: 'html' }) || '').trim();
                text = (this._tinymce.getContent({ format: 'text' }) || '').replace(/\u00A0/g, ' ').trim();
            } else {
                const ta = this.$el.querySelector('#patientNotesInput');
                const raw = (ta && ta.value ? ta.value : '').trim();
                text = raw || html.replace(/<[^>]*>/g, ' ').replace(/&nbsp;/gi, ' ').trim();
            }

            if (!text.length) {
                this.$set(this.errors, 'general', 'Please enter a note.');
                return;
            }

            this.loading = true;

            // IMPORTANT: send BOTH keys — backend often validates "note"
            const payload = {
                id: this.form.id || null,
                note: text,    // plain text (what the validator likely checks)
                content: html  // keep HTML formatting, if backend stores it
            };

            this.$http.post(this.routes['admin-member-notes-store'], payload)
                .then(({ data }) => {
                    this.loading = false;

                    if (data && data.errors) {
                        this.errors = data.errors || {};
                        return;
                    }

                    this.resetForm();
                    this.clearEditor();
                    this.$parent.emitEvent('newNoteAdded');
                    this.close();
                })
                .catch(() => {
                    this.loading = false;
                    this.$set(this.errors, 'general', 'Unable to save note. Please try again.');
                });
        },

        emitEvent(name, data) {
            this.$parent.emitEvent(name, data);
        }
    }
});
