Vue.component('patientWeightTrack', {
    props: ['routes','events','parent'],

    data() {
        return {
            records: [],
            loading: false,
            add: false,
            errors: {},
            success: false,
            display: 5,
            defaultWeightRecord: {
                id: null,
                metric: 1,
                heightCm: '',
                heightFt: '',
                heightIn: '',
                weightKg: '',
                weightSt: '',
                weightLb: '',
            },
            record: {},
            deleteIndex: -1,
            todayDate: null,
            showingAll: false,
            eventIndex: 0,
            reviewID: null,
        };
    },

    mounted() {
        this.init();
    },

    watch: {
        events(stack) {
            for (let i = this.eventIndex; i < stack.length; i++) {
                const event = stack[i];
                if (event.name === 'initReview') {
                    this.reviewID = event.data.reviewId;
                    this.load();
                }
                if (event.name === 'weightTrackUpdated') {
                    this.load();
                }
            }
            this.eventIndex = stack.length;
        }
    },

    computed: {
        canShowAll() {
            const list = Array.isArray(this.records) ? this.records : [];
            if (!list.length) return false;
            if (this.showingAll) return false;
            return list.length > this.display;
        }
    },

    methods: {
        init() {
            this.record = { ...this.defaultWeightRecord };
            if (this.parent !== 'review') {
                this.load();
            }
        },

        // cm → [ft, in]
        cmToFtIn(cm) {
            const n = Number(cm);
            if (!isFinite(n) || n <= 0) return ['', ''];
            const totalIn = n / 2.54;
            const ft = Math.floor(totalIn / 12);
            const inch = Math.round(totalIn - ft * 12);
            return [String(ft), String(inch)];
        },

        load() {
            this.loading = true;

            const params = (this.parent === 'review' && this.reviewID > 0) ? { review_id: this.reviewID } : {};
            const vm = this;

            this.$http.post(this.routes['admin-patient-weight-track-load'], params)
                .then((response) => {
                    let payload = response.data;

                    // vue-resource may return JSON as a string
                    if (typeof payload === 'string') {
                        try { payload = JSON.parse(payload); }
                        catch (e) { console.error('Invalid JSON from weight-track-load:', e, payload); payload = {}; }
                    }

                    const list       = Array.isArray(payload?.weightTrack) ? payload.weightTrack : [];
                    const today      = payload?.todayDate ?? null;
                    const heightCm   = Number(payload?.height ?? 0) || 0;
                    let heightImp    = Array.isArray(payload?.heightImperial) ? payload.heightImperial : null;
                    if (!heightImp || heightImp.length < 2) {
                        heightImp = vm.cmToFtIn(heightCm);
                    }

                    vm.records = list;                         // always an array
                    vm.todayDate = today;
                    vm.defaultWeightRecord = {
                        ...vm.defaultWeightRecord,
                        heightCm: heightCm ? String(heightCm) : '',
                        heightFt: heightImp[0] ?? '',
                        heightIn: heightImp[1] ?? '',
                    };
                    vm.loading = false;
                })
                .catch((err) => {
                    console.error('Weight track load failed:', err);
                    this.loading = false;
                    this.records = []; // keep it an array
                });
        },

        addRecord() {
            this.add = true;
            this.record = { ...this.defaultWeightRecord };
            $('#weightTrackDate' + this.parent).val(this.todayDate || '');
        },

        cancel() {
            this.add = false;
        },

        store() {
            if (this.loading) return;
            this.errors = {};
            this.success = false;
            this.loading = true;

            this.record.date_provided = $('#weightTrackDate' + this.parent).val();
            this.record.review_id = this.reviewID;

            const vm = this;

            this.$http.post(this.routes['admin-patient-weight-track-store'], this.record)
                .then((response) => {
                    let payload = response.data;
                    if (typeof payload === 'string') {
                        try { payload = JSON.parse(payload); } catch { payload = {}; }
                    }

                    if (payload.errors) {
                        vm.errors = payload.errors;
                        vm.loading = false;
                        return;
                    }
                    if (payload.success) {
                        vm.success = payload.success;
                    }
                    vm.record = { ...vm.defaultWeightRecord };
                    vm.add = false;
                    vm.emitEvent('weightTrackUpdated');
                })
                .catch((err) => {
                    console.error('Weight track store failed:', err);
                    this.loading = false;
                });
        },

        showConfirmDeletion(index) {
            this.deleteIndex = index;
        },

        deleteRecord(id) {
            if (this.loading) return;
            this.errors = {};
            this.success = false;
            this.loading = true;

            const vm = this;

            this.$http.post(this.routes['admin-patient-weight-track-delete'], { id })
                .then((response) => {
                    let payload = response.data;
                    if (typeof payload === 'string') {
                        try { payload = JSON.parse(payload); } catch { payload = {}; }
                    }

                    if (payload.errors) {
                        vm.errors = payload.errors;
                        vm.loading = false;
                        return;
                    }
                    if (payload.success) {
                        vm.success = payload.success;
                    }
                    vm.showConfirmDeletion(-1);
                    vm.emitEvent('weightTrackUpdated');
                })
                .catch((err) => {
                    console.error('Weight track delete failed:', err);
                    this.loading = false;
                });
        },

        showAll() {
            this.showingAll = true;
        },

        emitEvent(name, data) {
            this.$parent.emitEvent(name, data);
        },
    },
});
