<?php
/**
 * Brand Object Eloquent model
 *
 * @version 16/09/16
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Shop;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use MtcPharmacy\Multisite\Classes\MultisiteManager;
use MtcPharmacy\Multisite\Classes\MultisiteConstants;
use MtcPharmacy\Multisite\Classes\MultisiteSite;
use Spatie\Sitemap\Contracts\Sitemapable;
use Spatie\Sitemap\Tags\Url;

/**
 * Brand Object Eloquent model
 *
 * @version 16/09/16
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class Brand extends Model implements Sitemapable
{
    /**
     * @var string Model table name
     */
    protected $table = 'brands';

    /**
     * @var array|bool DateTime columns for object
     */
    public $timestamps = false;

    /**
     * @var array The attributes that are mass assignable.
     */
    protected $fillable = [
        'order',
        'name',
        'description',
        'hide',
        'sex',
        'seo_title',
        'seo_keywords',
        'seo_description'
    ];

    /**
     * Scope - active()
     * Discards all hidden and deleted brands
     *
     * @param \Illuminate\Database\Eloquent\Builder $query Query builder object
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeActive($query)
    {
        return $query->where('hide', 0)
            ->where('deleted', 0);
    }

    /**
     * Scope - selectBasics()
     * Retrieves basic info about brand - id & name
     *
     * @param \Illuminate\Database\Eloquent\Builder $query Query builder object
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeSelectBasics($query)
    {
        return $query->select([
            'brands.id',
            'brands.name'
        ]);
    }

    /*
     * Items relationship.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function items()
    {
        return $this->belongsToMany(Item::class, 'items_brands', 'brand_id');
    }

    /**
     * Get all active brands ordered by order and name.
     * This function generates a URL for all returned brands
     *
     * @return array
     */
    public static function getAllBrands() {
        $query = Brand::active()
            ->selectBasics()
            ->orderBy('order', 'ASC')
            ->orderBy('name', 'ASC');

        MultisiteManager::decorateQueryBaseOnly($query, MultisiteConstants::ENTITY_TYPE_SHOP_BRAND);

        return $query->get()
            ->map(function ($brand) {
                $brand['url'] = browse_url(0, $brand->id);
                return $brand;
            })
            ->toArray();
    }


    public function replicateWithRelations()
    {
        $clone = $this->replicate();
        $clone->save();

        return $clone;
    }


    public function getAdminUrl()
    {
        $url = "/shop/admin/brands/edit.brands.php?id={$this->id}";

        return $url;
    }


    public static function getExportableFields()
    {
        return [
            'brands.id',
            'brands.order',
            'brands.name',
            'brands.description',
            'brands.hide',
            'brands.seo_title',
            'brands.seo_keywords',
            'brands.seo_description',
        ];
    }


    public function getUrlAttribute()
    {
        return browse_url(0, $this->id);
    }


    public function toSitemapTag(): Url | string | array
    {
        $site_config = MultisiteSite::getSiteConfig($this->sitemap_site);

        return Url::create($site_config['SITE_URL'] . $this->url)
            ->setLastModificationDate(Carbon::create($this->updated))
            ->setChangeFrequency(Url::CHANGE_FREQUENCY_MONTHLY)
            ->setPriority(0.6)
            ;
    }

}
