# Component Library Documentation

**Last Updated**: 2025-10-06
**Branch**: `improvement/create-elements`
**Components**: 1 component (button)
**Note**: Form elements (inputs, selects, textareas) handled by existing `form_field()` function in `core/templates/includes/input_fields/`

---

## Overview

This directory contains a reusable Twig component for the pharmacy starter project: the button component.

- **`elements/button.twig`**: Button and link component with variants
- **Form elements**: Handled by `form_field()` PHP function in `core/templates/includes/input_fields/`
- **Other elements**: Existing templates in `includes/` (tooltip, payment icons, etc.)

---

## Component Hierarchy

```
elements/
└── button.twig   # Buttons and links (NEW - actively used)

partials/         # Existing composite components
page_blocks/      # Page sections
page_structure/   # Layout templates
```

**Note**: Form inputs (text, email, select, textarea, radio, checkbox) are handled via the `form_field()` PHP function in `core/templates/includes/input_fields/`, not as Twig components.

---

## Component Usage

### Button Component

**File**: `elements/button.twig`

#### Parameters

| Parameter | Type | Required | Default | Description |
|-----------|------|----------|---------|-------------|
| `text` | string | ⚠️ Conditional | - | Button text/label (required unless `icon` provided) |
| `href` | string | No | - | URL (renders `<a>` instead of `<button>`) |
| `type` | string | No | `'button'` | Button type (`button`, `submit`, `reset`) |
| `name` | string | No | - | Form field name (for identifying submit buttons) |
| `variant` | string | No | - | Style variant (`primary`, `alt`, `outlined`, `link`, `large`) |
| `icon` | string | No | - | Icon class name |
| `icon_position` | string | No | `'left'` | Icon position (`left`, `right`) |
| `target` | string | No | - | Link target attribute (`_blank`, `_self`, etc.) |
| `disabled` | boolean | No | `false` | Disable button |
| `additional_class` | string | No | - | Extra CSS classes |
| `data_attrs` | object | No | `{}` | Data attributes (`{'no-ajax': 'true'}`) |

#### Behavior Rules

1. **Element Type Logic**:
   - If `href` provided → renders `<a>`
   - If no `href` → renders `<button>`
   - If both `href` and `type` → `href` takes precedence

2. **CSS Class Construction**:
   - Base class: `button` (always applied)
   - Variant class: `button{Variant}` (e.g., `buttonPrimary`, `buttonOutlined`)
   - Additional classes appended

3. **Default Behavior**:
   - Missing `text` and `icon` → element hidden (renders nothing)
   - Icon-only buttons supported (text can be empty if icon provided)
   - Missing `type` → defaults to `'button'`
   - `name` attribute only renders for `<button>` elements

#### Usage Examples

```twig
{# Simple button #}
{% include 'includes/elements/button.twig' with {
    text: 'Submit'
} only %}
{# Output: <button type="button" class="button">Submit</button> #}

{# Link button with variant #}
{% include 'includes/elements/button.twig' with {
    text: 'View More',
    href: '/products',
    variant: 'outlined'
} only %}
{# Output: <a href="/products" class="button buttonOutlined">View More</a> #}

{# Submit button with icon #}
{% include 'includes/elements/button.twig' with {
    text: 'Save Changes',
    type: 'submit',
    icon: 'icon-save',
    icon_position: 'left',
    variant: 'primary'
} only %}

{# Button with data attributes #}
{% include 'includes/elements/button.twig' with {
    text: 'Add to Cart',
    additional_class: 'mtcOverlay',
    data_attrs: {
        'no-ajax': 'true',
        'product-id': product.id
    }
} only %}
{# Output: <button ... data-no-ajax="true" data-product-id="123"> #}

{# Disabled button #}
{% include 'includes/elements/button.twig' with {
    text: 'Out of Stock',
    disabled: true,
    variant: 'alt'
} only %}

{# Link opening in new tab #}
{% include 'includes/elements/button.twig' with {
    text: 'External Link',
    href: 'https://example.com',
    target: '_blank',
    variant: 'link'
} only %}

{# Submit button with name attribute #}
{% include 'includes/elements/button.twig' with {
    text: 'Send message',
    type: 'submit',
    name: 'trigger',
    variant: 'alt'
} only %}
{# Output: <button type="submit" name="trigger" class="button buttonAlt">Send message</button> #}

{# Icon-only button (for inline actions) #}
{% include 'includes/elements/button.twig' with {
    text: '',
    icon: 'far fa-long-arrow-right',
    type: 'submit',
    name: 'signup',
    additional_class: 'btn'
} only %}
{# Output: <button type="submit" name="signup" class="button btn"><i class="far fa-long-arrow-right"></i></button> #}

{# Button with complex name (array syntax) #}
{% include 'includes/elements/button.twig' with {
    text: 'Save Changes',
    type: 'submit',
    name: "submit['submit_health_profile']",
    variant: 'alt'
} only %}
```

---

## Migration Patterns

### Before: Inline Button Markup
```twig
<a href="{{ url }}" class="button buttonOutlined">{{ text }}</a>
```

### After: Button Component
```twig
{% include 'includes/elements/button.twig' with {
    text: text,
    href: url,
    variant: 'outlined'
} only %}
```

---

## Best Practices

### 1. Always Use `only` Keyword
```twig
{# ✅ GOOD: Isolates component scope #}
{% include 'includes/elements/button.twig' with {
    text: 'Click Me'
} only %}

{# ❌ BAD: Component can access parent template variables #}
{% include 'includes/elements/button.twig' with {
    text: 'Click Me'
} %}
```

### 2. Use Named Parameters
```twig
{# ✅ GOOD: Clear, readable #}
{% include 'includes/elements/button.twig' with {
    text: 'Submit',
    type: 'submit',
    variant: 'primary'
} only %}

{# ❌ BAD: Hard to understand #}
{% include 'includes/elements/button.twig' with button_data only %}
```

### 3. Provide Required Parameters
```twig
{# ✅ GOOD: All required params provided #}
{% include 'includes/elements/button.twig' with {
    text: 'Click Me'
} only %}

{# ❌ BAD: Missing required 'text' parameter - renders nothing #}
{% include 'includes/elements/button.twig' with {} only %}
```

### 4. Handle Empty Values
```twig
{# ✅ GOOD: Check before including #}
{% if product.cta_text is not empty %}
    {% include 'includes/elements/button.twig' with {
        text: product.cta_text,
        href: product.url
    } only %}
{% endif %}

{# ℹ️ ACCEPTABLE: Component handles empty text gracefully (renders nothing) #}
{% include 'includes/elements/button.twig' with {
    text: product.cta_text,
    href: product.url
} only %}
```

### 5. Use Ternary Operators for Conditional Parameters
```twig
{% include 'includes/elements/button.twig' with {
    text: product.in_stock ? 'Add to Cart' : 'Out of Stock',
    variant: product.in_stock ? 'primary' : 'alt',
    disabled: not product.in_stock
} only %}
```

---

## Troubleshooting

### Component Renders Nothing
**Cause**: Required parameter missing (e.g., `text` for button)
**Solution**: Check that all required parameters are provided and not empty

### Variant Classes Not Applying
**Cause**: Variant parameter misspelled or unsupported variant value
**Solution**: Check component documentation for valid variant options

---

## Testing Components

### Manual Testing
1. Browse to page using component
2. Verify visual appearance matches expected design
3. Test interaction (button click, form submission)
4. Check browser console for errors

### PHPUnit Testing
```bash
php artisan test tests/Unit/Components/ButtonComponentTest.php
```

---

## Component File Locations

**Component**:
- `sites/default/templates/includes/elements/button.twig`

**Test**:
- `tests/Unit/Components/ButtonComponentTest.php`

**Form Elements** (handled separately via `form_field()` function):
- `core/templates/includes/input_fields/text.twig`
- `core/templates/includes/input_fields/textarea.twig`
- `core/templates/includes/input_fields/select.twig`
- `core/templates/includes/input_fields/radio.twig`
- `core/templates/includes/input_fields/checkbox.twig`

---

## Migration Status

**Completed**:
- ✅ Button component: 29 instances migrated across 18 templates
  - Public-facing pages: 11 buttons (page blocks, product cards)
  - Member area: 13 buttons (messages, orders, reorder summary)
  - Forms with named buttons: 5 buttons (contact, newsletter, register, password reset, health profile)

**Remaining**:
- ~8 inline button instances (Vue.js templates and admin pages)
- Form elements (inputs, selects, textareas) - to be standardized using `form_field()` function in separate branch

**Note**: Badge and card components were removed (no usage found). Input component was removed to prevent duplication with existing `form_field()` PHP function (`core/templates/includes/input_fields/`).

See `specs/improvement-create-elements/migration-report.md` for detailed migration status.

---

*Documentation generated: 2025-10-06 | Component Library v1.0 | Branch: improvement/create-elements*
