<?php
/**
 * Class BaseImport
 *
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 * @contributor Uldis Zvirbulis <uldis.zvirbulis@mtcmedia.co.uk>
 * @version 04.09.2017
 */

namespace Mtc\Core\Import;

use Illuminate\Database\Capsule\Manager as Capsule;
use Illuminate\Support\Facades\DB;

/**
 * Class BaseImport
 *
 * Implements base import functionality
 *
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 * @contributor Uldis Zvirbulis <uldis.zvirbulis@mtcmedia.co.uk>
 * @version 04.09.2017
 */
class BaseImport
{
    protected $truncate = false;
    protected $import_members = false;
    protected $import_orders = false;
    protected $import_refunds = false;
    protected $import_categories = false;
    protected $import_products = false;
    protected $import_images = false;
    protected $import_forms = false;
    protected $import_assessments = false;
    protected $merge_member_duplicates = false;
    protected $run_patch = false;
    protected $image_folders = [];
    protected $db_host;
    protected $db_user;
    protected $db_pass;
    protected $db_name;

    /**
     * Base Import constructor.
     *
     * Each instance of import class must have a constructor that feeds in config.
     * Essential values:
     * - truncate - whether to truncate tables before import
     * - import_members - whether to import members from DB
     * - db_host - import DB hostname
     * - db_user - import DB username
     * - db_pass - import DB password
     * - db_name - import DB name
     *
     * @param array $config config of settings for import
     */
    public function __construct(Array $config)
    {
        // Assign config values
        foreach ($config as $key => $value) {
            $this->{$key} = $value;
        }

        // Define connection
        $this->setRemoteConnection();
    }

    /**
     * Create a new connection for the remote database
     *
     * @param string $host DB host
     * @param string $username DB Username
     * @param string $password DB Password
     * @param string $db_name DB Name
     */
    public function setRemoteConnection()
    {
        $capsule = new Capsule(\Illuminate\Support\Facades\App::getInstance());
        $capsule->addConnection([
            'driver'    => 'mysql',
            'host'      => $this->db_host,
            'database'  => $this->db_name,
            'username'  => $this->db_user,
            'password'  => $this->db_pass,
            'charset'   => 'utf8',
            'collation' => 'utf8_unicode_ci',
            'prefix'    => '',
            'strict'    => false,
        ], 'import');
    }

    /**
     * Do the full import process based on config
     * This will trigger all relevant sub-imports based on given config
     */
    public function import()
    {
        // To avoid running out of memory for larger DBs, disable query logs on both connections.
        DB::connection()->disableQueryLog();
        DB::connection('import')->disableQueryLog();

        if (MEMBERS_SECTION === true && $this->import_members) {
            $this->importMembers();
        }

        if ($this->import_categories) {
            $this->importCategories();
        }
        if ($this->import_products) {
            $this->importProducts();
        }
        if ($this->import_orders) {
            $this->importOrders();
        }
        if ($this->import_refunds) {
            $this->importRefunds();
        }
        if ($this->import_images) {
            $this->loadRemoteImages();
        }
        if ($this->import_forms) {
            $this->importForms();
        }
        if ($this->import_assessments) {
            $this->importAssessments();
        }
        if ($this->import_assessments) {
            $this->importAssessments();
        }

        if ($this->merge_member_duplicates) {
            $this->mergeMemberDuplicates();
        }

        if ($this->run_patch) {
            $this->patch();
        }
    }

}