<?php

namespace Mtc\PinnedItems\Tests\Unit;

use Illuminate\Foundation\Testing\DatabaseMigrations;

use Mtc\PinnedItems\Facades\PinnedItems;
use Mtc\PinnedItems\Models\PinnedItemModel;
use Mtc\PinnedItems\PinnedItemRepository;
use Mtc\PinnedItems\Tests\TestCase;

class PinnedItemRepositoryTest extends TestCase
{
    use DatabaseMigrations;

    public function testTrackingItem()
    {
        $repository = $this->app->make(PinnedItemRepository::class);
        $viewer_object = PinnedItemModel::query()
            ->create([]);

        $viewed_object = PinnedItemModel::query()
            ->create([]);

        $repository->add($viewer_object, $viewed_object, 'lorem ipsum');

        $was_added = PinnedItemModel::query()
            ->where('owner_id', $viewer_object->id)
            ->where('owner_type', $viewer_object->getMorphClass())
            ->where('pinned_item_id', $viewed_object->id)
            ->where('pinned_item_type', $viewed_object->getMorphClass())
            ->where('category', 'lorem ipsum')
            ->exists();

        self::assertTrue($was_added);
    }

    public function testRemovingItem()
    {
        $repository = $this->app->make(PinnedItemRepository::class);
        $viewer_object = PinnedItemModel::query()
            ->create([]);

        $viewed_object = PinnedItemModel::query()
            ->create([]);

        $repository->add($viewer_object, $viewed_object, 'lorem ipsum');

        $was_added = PinnedItemModel::query()
            ->where('owner_id', $viewer_object->id)
            ->where('owner_type', $viewer_object->getMorphClass())
            ->where('pinned_item_id', $viewed_object->id)
            ->where('pinned_item_type', $viewed_object->getMorphClass())
            ->where('category', 'lorem ipsum')
            ->exists();

        self::assertTrue($was_added);

        $repository->remove($viewer_object, $viewed_object, 'lorem ipsum');

        $was_added = PinnedItemModel::query()
            ->where('owner_id', $viewer_object->id)
            ->where('owner_type', $viewer_object->getMorphClass())
            ->where('pinned_item_id', $viewed_object->id)
            ->where('pinned_item_type', $viewed_object->getMorphClass())
            ->where('category', 'lorem ipsum')
            ->exists();

        self::assertFalse($was_added);
    }

    public function testRetrievingByCategory()
    {
        $repository = $this->app->make(PinnedItemRepository::class);
        $test_a = PinnedItemModel::query()->create([]);
        $test_b = PinnedItemModel::query()->create([]);
        $test_c = PinnedItemModel::query()->create([]);

        $repository->add($test_a, $test_b, 'lorem ipsum');
        $repository->add($test_a, $test_c, 'lorem ipsum');
        $repository->add($test_a, $test_b, 'foo baz bar');

        self::assertEquals(2, $repository->get($test_a, 'lorem ipsum')->count());
        self::assertEquals(1, $repository->get($test_a, 'foo baz bar')->count());
    }

    public function testRetrievingWithLimit()
    {
        $repository = $this->app->make(PinnedItemRepository::class);
        $test_a = PinnedItemModel::query()->create([]);
        $test_b = PinnedItemModel::query()->create([]);
        $test_c = PinnedItemModel::query()->create([]);
        $test_d = PinnedItemModel::query()->create([]);

        $repository->add($test_a, $test_b, 'lorem ipsum');
        $repository->add($test_a, $test_c, 'lorem ipsum');
        $repository->add($test_a, $test_d, 'lorem ipsum');
        $repository->add($test_a, $test_b, 'foo baz bar');

        self::assertEquals(3, $repository->get($test_a, 'lorem ipsum')->count());
        self::assertEquals(2, $repository->get($test_a, 'lorem ipsum', 2)->count());
    }

    public function testRetrievingWithoutCategory()
    {
        $repository = $this->app->make(PinnedItemRepository::class);
        $test_a = PinnedItemModel::query()->create([]);
        $test_b = PinnedItemModel::query()->create([]);
        $test_c = PinnedItemModel::query()->create([]);

        $repository->add($test_a, $test_b, 'lorem ipsum');
        $repository->add($test_a, $test_c, 'lorem ipsum');
        $repository->add($test_a, $test_b, 'foo baz bar');

        self::assertEquals(3, $repository->get($test_a)->count());
    }

    public function testAddingFromFacade()
    {
        $test_a = PinnedItemModel::query()->create([]);
        $test_b = PinnedItemModel::query()->create([]);

        PinnedItems::add($test_a, $test_b, 'lorem ipsum');

        self::assertEquals(1, PinnedItems::get($test_a, 'lorem ipsum')->count());
    }

    public function testCountWhenOwnerIsDeleted()
    {
        $repository = $this->app->make(PinnedItemRepository::class);
        $test_a = PinnedItemModel::query()->create([]);
        $test_b = PinnedItemModel::query()->create([]);
        $test_c = PinnedItemModel::query()->create([]);
        $test_d = PinnedItemModel::query()->create([]);

        $repository->add($test_a, $test_b, 'lorem ipsum');
        $repository->add($test_a, $test_c, 'lorem ipsum');
        $repository->add($test_a, $test_d, 'lorem ipsum');

        self::assertEquals(3, $repository->get($test_a, 'lorem ipsum')->count());

        $test_d->delete();
        self::assertEquals(2, $repository->get($test_a, 'lorem ipsum')->count());
    }
}