<?php

namespace MtcMedia\Queue;

use Debug\QueryLogFormatter;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Monolog\Logger;

class Queue extends Model
{
    /**
     * @var QueueManager
     */
    protected $manager;

    /**
     * @return QueueManager
     */
    public function getManager()
    {
        return $this->manager;
    }

    /**
     * @param QueueManager $manager
     */
    public function setManager(QueueManager $manager)
    {
        $this->manager = $manager;
    }

    public function addJob(Queueable $job)
    {
        $model = new JobModel();
        $model->name = $job->getName();
        $model->data = serialize($job);

        return $this->jobs()->save($model);
    }

    /**
     * Clear pending jobs of the same type and add this one
     * @param Queueable $job
     */
    public function addSingle(Queueable $job)
    {
        $this->jobs()->where('name', '=', $job->getName())
            // if a job has been attempted, it might be running just now, so we won't delete it
            ->where('attempts', 0)
            ->delete()
        ;

        $this->addJob($job);
    }

    public function work($number = 120)
    {
        while ($number-- && ($job_model = $this->nextJob())) {
            $this->runJob($job_model);
        }
    }

    public function runJob(JobModel $job_model)
    {
        $job = $job_model->getJob();

        // mark the job as attempted
        if ($job_model->attempt() === 0) {
            // if no records upated in DB, the job must have been deleted, so we ignore the job
            return;
        }

        try {
            // run the job
            $result = $job->execute();

            // validate the result
            if (!is_bool($result)) {
                throw new \RuntimeException("Queue job execution result should be a boolean value");
            }

            // if job has run successfully, mark the job as 'done'
            if ($result) {
                $job_model->done();
            } else {
                // @todo log failed attempt
            }
        } catch (\Exception $e) {
            $subject = sprintf("[%s] Exception when running a queued job [%s]", SITE_NAME, $job->getName());
            $body = [
                'message' => $e->getMessage(),
                'file' => $e->getFile() . ': ' . $e->getLine(),
                'trace' => $e->getTraceAsString(),
            ];
            $body_as_string = print_r($body, true);
            if (($logger = $this->manager->getLogger()) instanceof Logger) {
                $logger->addAlert($subject, $body);
            } else {
                mail(DEV_EMAIL, $subject, $body_as_string);
            }

            # doesn't work in the browser
            #file_put_contents(STDERR, $subject . "\n" . $body_as_string);
            if (ENVIRONMENT !== 'live') {
                throw $e;
            }
        }
    }

    // RELATIONSHIPS

    /**
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function jobs()
    {
        return $this->hasMany(JobModel::class);
    }

    public function queuedJobs()
    {

    }

    /**
     * @return JobModel
     */
    private function nextJob()
    {
        return $this->jobs()->pending()
            // fetching jobs one by one to prevent outated data in the fetched queue jobs
            ->take(1)
            ->orderBy('attempts')
            ->orderBy('created_at')
            ->first()
        ;
    }
}
