<?php

namespace Mtc\Realex;

use Exception;
use GlobalPayments\Api\Entities\Enums\HppVersion;
use GlobalPayments\Api\Entities\HostedPaymentData;
use GlobalPayments\Api\HostedPaymentConfig;
use GlobalPayments\Api\Services\HostedService;
use GlobalPayments\Api\ServicesConfig;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\PayableContract;
use Mtc\Checkout\Contracts\PaymentGateway;
use Mtc\Checkout\PaymentForm;
use Mtc\Modules\Members\Classes\Auth;
use Mtc\Modules\Members\Classes\Member;

/**
 * Realex Payment Gateway
 *
 * @package  Mtc\Realex
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class Realex implements PaymentGateway
{
    /**
     * Check if the gateway is available for use on this payment.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return bool
     */
    public function isApplicable(InvoiceRepositoryContract $invoice, $payable): bool
    {
        if ($invoice->getOutstandingAmount() <= 0.01) {
            return false;
        }

        return App::make(config('realex.applicable_check_class'))->handle($invoice, $payable);
    }

    /**
     * Render the payment template.
     *
     * @param InvoiceRepositoryContract $invoice
     * @return PaymentForm
     */
    public function getPaymentForm(InvoiceRepositoryContract $invoice): PaymentForm
    {
        $member = Auth::getLoggedInMember();
        if ($member->exists && config('realex.use_card_storage')) {
            $customer_details = (new Customer)->findCustomer($member);
        }

        $service = new HostedService($this->loadConfig($member, $customer_details, $invoice->getCurrency()));
        $payment_data = new HostedPaymentData();

        if ($member->exists && config('realex.use_card_storage')) {
            $payment_data->offerToSaveCard = true;
            $payment_data->customerExists = !empty($customer_details['customer_token']);
            if (!empty($customer_details['customer_token'])) {
                $payment_data->customerKey = $customer_details['customer_token'];
                $payment_data->paymentKey = $customer_details['payment_token'];
            }
        }

        if (config('realex.multi_currency_enabled')) {
            $json = $service->charge($invoice->getOutstandingAmountInCurrency())
                ->withCurrency($invoice->getCurrency())
                ->withHostedPaymentData($payment_data)
                ->serialize();
        } else {
            $json = $service->charge($invoice->getOutstandingAmount())
                ->withCurrency(config('currencies.default_currency'))
                ->withHostedPaymentData($payment_data)
                ->serialize();
        }

        return new PaymentForm('realex-payment', 'vue-component', [
            'gateway_url' => $this->getApiEndpoint(),
            'pay_url' => route('charge_payment', [ $invoice->getId() ]) . '?gateway=realex',
            'merchant_id' => config('realex.merchant_id'),
            'account_id' => config('realex.account_id'),
            'show_test_cards' => !config('realex.use_production_endpoint', false),
            'json' => $json,
            'name' => __('realex::realex.payment_option_name')
        ]);
    }

    /**
     * Charge payment on invoice
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return bool
     * @throws Exception
     */
    public function charge(Request $request, InvoiceRepositoryContract $invoice): array
    {
        $member = $invoice->getMember();
        if ($member->exists && config('realex.use_card_storage')) {
            $customer_details = (new Customer)->findCustomer($member);
        }

        $service = new HostedService($this->loadConfig($member, $customer_details, $invoice->getCurrency()));

        // create the response object from the response
        // Since it expects JSON response but we receive post data we need to encode it
        $parsed_response = $service->parseResponse(json_encode($request->input()));

        if ($parsed_response->responseCode !== '00') {
            throw new Exception($parsed_response->responseMessage);
        }

        if ($member->exists) {
            (new Customer)->storeCustomerDetails($member, $parsed_response);
        }

        return [
            'provider' => 'realex',
            'amount' => $invoice->getOutstandingAmount(),
            'currency_code' => $invoice->getCurrency(),
            'amount_in_currency' => $parsed_response->responseValues['AMOUNT'] / 100,
            'reference' => $parsed_response->orderId,
            'details' => $parsed_response->responseValues,
            'confirmed_at' => now(),
        ];
    }

    /**
     * Get the payment endpoint for the gateway
     *
     * @return string
     */
    public function getApiEndpoint(): string
    {
        return config('realex.use_production_endpoint', false)
            ? 'https://pay.realexpayments.com/pay'
            : 'https://pay.sandbox.realexpayments.com/pay';
    }

    /**
     * Initialize config for payment
     *
     * @param Member $member
     * @param array $member_details
     * @param string $currency
     * @return ServicesConfig
     */
    protected function loadConfig($member, $member_details, $currency): ServicesConfig
    {
        $config = new ServicesConfig();
        $config->merchantId = config('realex.merchant_id');
        $config->accountId = $this->getAccountId($currency);
        $config->sharedSecret = config('realex.secret');
        $config->serviceUrl = $this->getApiEndpoint();
        $config->hostedPaymentConfig = new HostedPaymentConfig();
        $config->hostedPaymentConfig->version = HppVersion::VERSION_2;
        if ($member->exists && config('realex.use_card_storage')) {
            if (!empty($member_details['customer_token'])) {
                $config->hostedPaymentConfig->displaySavedCards = true;
            } else {
                $config->hostedPaymentConfig->cardStorageEnabled = '1';
            }
        }
        return $config;
    }

    /**
     * Find the relevant account ID based on selected currency
     *
     * @param $currency
     * @return \Illuminate\Config\Repository|mixed
     */
    protected function getAccountId($currency)
    {
        if (config('realex.multi_currency_enabled')) {
            return config("realex.currency_specific_account_ids.{$currency}") ?: config('realex.account_id');
        }

        return config('realex.account_id');
    }
}
