<?php

namespace Mtc\RecentlyViewed\Tests\Unit;

use Illuminate\Foundation\Testing\DatabaseMigrations;

use Mtc\RecentlyViewed\Facades\RecentlyViewed;
use Mtc\RecentlyViewed\Models\RecentlyViewedModel;
use Mtc\RecentlyViewed\RecentlyViewedRepository;
use Mtc\RecentlyViewed\Tests\TestCase;

class RecentlyViewedRepositoryTest extends TestCase
{
    use DatabaseMigrations;

    public function testTrackingItem()
    {
        $repository = $this->app->make(RecentlyViewedRepository::class);
        $viewer_object = RecentlyViewedModel::query()
            ->create([]);

        $viewed_object = RecentlyViewedModel::query()
            ->create([]);

        $repository->track($viewer_object, $viewed_object, 'lorem ipsum');

        $was_added = RecentlyViewedModel::query()
            ->where('viewer_id', $viewer_object->id)
            ->where('viewer_type', $viewer_object->getMorphClass())
            ->where('viewed_id', $viewed_object->id)
            ->where('viewed_type', $viewed_object->getMorphClass())
            ->where('category', 'lorem ipsum')
            ->exists();

        self::assertTrue($was_added);
    }

    public function testRetrievingByCategory()
    {
        $repository = $this->app->make(RecentlyViewedRepository::class);
        $test_a = RecentlyViewedModel::query()->create([]);
        $test_b = RecentlyViewedModel::query()->create([]);
        $test_c = RecentlyViewedModel::query()->create([]);

        $repository->track($test_a, $test_b, 'lorem ipsum');
        $repository->track($test_a, $test_c, 'lorem ipsum');
        $repository->track($test_a, $test_b, 'foo baz bar');

        self::assertEquals(2, $repository->get($test_a, 'lorem ipsum')->count());
        self::assertEquals(1, $repository->get($test_a, 'foo baz bar')->count());
    }

    public function testRetrievingWithLimit()
    {
        $repository = $this->app->make(RecentlyViewedRepository::class);
        $test_a = RecentlyViewedModel::query()->create([]);
        $test_b = RecentlyViewedModel::query()->create([]);
        $test_c = RecentlyViewedModel::query()->create([]);
        $test_d = RecentlyViewedModel::query()->create([]);

        $repository->track($test_a, $test_b, 'lorem ipsum');
        $repository->track($test_a, $test_c, 'lorem ipsum');
        $repository->track($test_a, $test_d, 'lorem ipsum');
        $repository->track($test_a, $test_b, 'foo baz bar');

        self::assertEquals(3, $repository->get($test_a, 'lorem ipsum')->count());
        self::assertEquals(2, $repository->get($test_a, 'lorem ipsum', 2)->count());
    }

    public function testRetrievingWithoutCategory()
    {
        $repository = $this->app->make(RecentlyViewedRepository::class);
        $test_a = RecentlyViewedModel::query()->create([]);
        $test_b = RecentlyViewedModel::query()->create([]);
        $test_c = RecentlyViewedModel::query()->create([]);

        $repository->track($test_a, $test_b, 'lorem ipsum');
        $repository->track($test_a, $test_c, 'lorem ipsum');
        $repository->track($test_a, $test_b, 'foo baz bar');

        self::assertEquals(3, $repository->get($test_a)->count());
    }

    public function testDroppingOlderEntries()
    {
        $repository = new RecentlyViewedRepository(['history_per_category' => 3]);
        $test_a = RecentlyViewedModel::query()->create([]);
        $test_b = RecentlyViewedModel::query()->create([]);
        $test_c = RecentlyViewedModel::query()->create([]);
        $test_d = RecentlyViewedModel::query()->create([]);

        $repository->track($test_a, $test_b, 'lorem ipsum');
        $repository->track($test_a, $test_c, 'lorem ipsum');
        $repository->track($test_a, $test_b, 'lorem ipsum');
        $repository->track($test_a, $test_d, 'lorem ipsum');
        $repository->track($test_a, $test_d, 'foo baz bar');

        self::assertEquals(3, $repository->get($test_a, 'lorem ipsum')->count());
        self::assertEquals(1, $repository->get($test_a, 'foo baz bar')->count());
    }

    public function testTrackingExistingEntryUpdatesRecord()
    {
        $repository = $this->app->make(RecentlyViewedRepository::class);
        $test_a = RecentlyViewedModel::query()->create([]);
        $test_b = RecentlyViewedModel::query()->create([]);
        $test_c = RecentlyViewedModel::query()->create([]);
        $test_d = RecentlyViewedModel::query()->create([]);

        $repository->track($test_a, $test_b, 'lorem ipsum');
        $repository->track($test_a, $test_c, 'lorem ipsum');
        $repository->track($test_a, $test_d, 'foo baz bar');

        $repository->track($test_a, $test_b, 'lorem ipsum');
        $viewed = $repository->get($test_a);

        self::assertEquals($test_b->id, $viewed->first()->id);

        sleep(1);
        $repository->track($test_a, $test_c, 'lorem ipsum');
        $viewed = $repository->get($test_a);

        self::assertEquals($test_c->id, $viewed->first()->id);
    }

    public function testAddingFromFacade()
    {
        $test_a = RecentlyViewedModel::query()->create([]);
        $test_b = RecentlyViewedModel::query()->create([]);

        RecentlyViewed::track($test_a, $test_b, 'lorem ipsum');

        self::assertEquals(1, RecentlyViewed::get($test_a, 'lorem ipsum')->count());
    }
}