<?php

namespace Mtc\Refunds;

use Carbon\Carbon;
use Illuminate\Support\Facades\Log;
use Mtc\Checkout\Contracts\HandlesRefunds;
use Mtc\Checkout\Facades\Payment as PaymentProcessor;
use Mtc\Checkout\Invoice\Payment;
use Mtc\Checkout\Facades\Payment as Payments;

/**
 * Class RefundManager
 *
 * @package Mtc\Refunds
 */
class RefundManager
{
    /**
     * Check if payment can be refunded
     *
     * @param Payment $payment
     * @return bool
     */
    public function canRefundPayment(Payment $payment)
    {
        return $this->paymentProviderSupportsRefunds($payment->provider)
            && $this->paymentHasNotExpired($payment)
            && $this->paymentIsNotRefunded($payment);
    }

    /**
     * Handle a refund
     *
     * @param Payment $payment
     * @return bool|array
     */
    public function refundPayment(Payment $payment, $amount)
    {
        PaymentProcessor::setActiveDriver($payment->provider);
        try {
            return PaymentProcessor::refund($payment, $amount);
        } catch (\Exception $exception) {
            Log::error('Refund failed', [
                'payment' => $payment,
                'exception' => $exception,
            ]);
            return false;
        }
    }

    /**
     * Check if payment provider has ability to process refunds
     *
     * @param $provider
     * @return bool
     */
    protected function paymentProviderSupportsRefunds($provider): bool
    {
        Payments::setActiveDriver($provider);
        return Payments::supports(HandlesRefunds::class);
    }

    /**
     * Check if payment has not expired
     *
     * @param $payment
     * @return bool
     */
    protected function paymentHasNotExpired($payment): bool
    {
        return $payment->confirmed_at->diffInDays(Carbon::now()) < config('refunds.expiry_days');
    }

    /**
     * Check if payment has not expired
     *
     * @param $payment
     * @return bool
     */
    protected function paymentIsNotRefunded($payment): bool
    {
        Payments::setActiveDriver($payment->provider);
        return Payments::isRefundable($payment);
    }
}
