<?php

namespace Mtc\Reports\Contracts;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Config;
use Mtc\Reports\Factories\ReportLogFactory;

abstract class ReportLogModel extends Model
{
    use HasFactory;

    const MAX_LOGS = 30;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'report_logs';

    /**
     * @var string[]
     */
    protected $guarded = [
        'id'
    ];

    /**
     * Boot the model and add a created event listener to enforce the log limit.
     */
    protected static function boot()
    {
        parent::boot();

        // Trigger on the created event
        static::created(function (ReportLogModel $log) {
            $log->enforceLogLimit();
        });
    }

    /**
     * Define model factory
     *
     * @return ReportLogFactory
     */
    protected static function newFactory()
    {
        return ReportLogFactory::new();
    }

    /**
     * Relationship with the report model
     *
     * @return BelongsTo
     */
    public function report(): BelongsTo
    {
        return $this->belongsTo(Config::get('reports.report_model'));
    }

    /**
     * Enforce the maximum number of logs per report.
     */
    protected function enforceLogLimit(): int
    {
        $excessLogs = self::query()
            ->where('report_id', $this->report_id)
            ->orderBy('created_at', 'desc')
            ->pluck('id')
            ->slice(self::MAX_LOGS);

        return self::whereIn('id', $excessLogs)->delete();
    }
}
