<?php

namespace Mtc\Reports\Contracts;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Config;
use Mtc\Reports\Factories\ReportFactory;

abstract class ReportModel extends Model
{
    use HasFactory;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'reports';

    /**
     * @var string[]
     */
    protected $guarded = [
        'id'
    ];

    protected $casts = [
        'active' => 'boolean',
        'columns' => 'array',
        'conditions' => 'array',
        'created_at' => 'datetime',
        'last_report_sent_at' => 'datetime:Y-m-d H:i',
        'next_report_due_at' => 'datetime:Y-m-d H:i',
    ];

    /**
     * Define model factory
     *
     * @return ReportFactory
     */
    protected static function newFactory()
    {
        return ReportFactory::new();
    }

    /**
     * Relationship with report logs
     *
     * @return HasMany
     */
    public function logs(): HasMany
    {
        return $this->hasMany(Config::get('reports.log_model'));
    }

    /**
     * Scope to filter by active state
     *
     * @param Builder $query
     * @param $value
     * @return void
     */
    public function scopeActive(Builder $query, $value = null)
    {
        if (in_array($value, [0, 1], true)) {
            $query->where('active', $value);
        }
    }

    /**
     * Get the next scheduled datetime for the report
     *
     * @param Carbon $from
     * @return mixed
     */
    public function getNextDueAt(Carbon $from)
    {
        return App::make(Config::get('reports.schedule_types.' . $this->schedule))->nextTime($from);
    }

    /**
     * Check if report should be sent as an attachment file or in-email
     *
     * @return bool
     */
    public function sendAsFile(): bool
    {
        return array_key_exists($this->format, Config::get('reports.formats_as_attachments', []));
    }

    /**
     * Return the export class
     *
     * @return string
     */
    public function exportClass(): string
    {
        return Config::get('reports.formats_as_attachments.' . $this->format);
    }
}
