<?php

namespace Mtc\Returns;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Event;
use Mtc\Orders\Contracts\OrderContract;
use Mtc\Returns\Events\NewReturn;
use Mtc\Returns\Events\ReturnCancelled;
use Mtc\Returns\Events\ReturnProcessed;
use Mtc\Returns\Http\Presenters\ReturnRequestUrlPresenter;
use Mtc\Returns\Scopes\ExcludesDraft;

/**
 * Class ReturnRequest
 *
 * @property $refunded_total
 *
 * @package Mtc\Returns
 */
class ReturnRequest extends Model
{
    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'returns';

    /**
     * Mass assignable attributes
     *
     * @var array
     */
    protected $fillable = [
        'status_id',
        'order_id',
        'reference',
        'message',
        'delivery_refund_amount',
        'locked',
    ];

    /**
     * Cast attributes to certain types
     *
     * @var array
     */
    protected $casts = [
        'locked' => 'boolean',
        'created_at' => 'datetime',
    ];

    /**
     * Boot the model
     */
    protected static function boot()
    {
        parent::boot();
        self::addGlobalScope(new ExcludesDraft());

        self::saved(function (self $request) {
            $request->triggerStatusEvents();
        });

        self::created(function (self $request) {
            $request->createEvent();
        });
    }

    /**
     * @param Builder $query
     * @param Request $request
     * @return Builder
     */
    public function scopeSearch(Builder $query, Request $request)
    {
        collect(config('returns.admin_search_filters', []))
            ->each(function ($filter_class) use ($request, $query) {
                App::make($filter_class)->handle($request, $query);
            });

        return $query;
    }

    /**
     * Items in this return request
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function items()
    {
        return $this->hasMany(ReturnItem::class, 'return_id');
    }

    /**
     * History of actions
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function history()
    {
        return $this->hasMany(ReturnHistory::class, 'return_id')
            ->latest();
    }

    /**
     * Order for which a return has been submitted
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function order()
    {
        return $this->belongsTo(config('orders.order_class'));
    }

    /**
     * Get the status name
     *
     * @return array|string|null
     */
    public function getStatusAttribute()
    {
        $status_name = array_search($this->status_id, config('returns.statuses', []));
        return __("returns::returns.statuses.{$status_name}");
    }

    /**
     * Get the status name
     *
     * @return array|string|null
     */
    public function getStatusBadgeAttribute()
    {
        $status_name = array_search($this->status_id, config('returns.statuses', []));
        return config("returns.status_badges.{$status_name}");
    }

    /**
     * Use Presenter to set up url attribute
     *
     * @return ReturnRequestUrlPresenter
     */
    public function getUrlAttribute()
    {
        return new ReturnRequestUrlPresenter($this);
    }

    /**
     * Get the list of status names
     *
     * @return \Illuminate\Support\Collection
     */
    public static function statusNames()
    {
        return collect(array_flip(config('returns.statuses', [])))
            ->map(function ($status_name) {
                return __('returns::returns.statuses.' . $status_name);
            });
    }

    /**
     * Trigger any status change events required
     */
    public function triggerStatusEvents()
    {

        if ($this->isDirty('status_id') == false) {
            return;
        }

        switch ($this->status_id) {
            case config('returns.statuses.processed'):
                $this->processedEvent();
                break;

            case config('returns.statuses.cancelled'):
                $this->cancelEvent();
                break;
        }
    }

    /**
     * Mark return as created
     */
    public function createEvent()
    {
        Event::dispatch(new NewReturn($this));
    }
    /**
     * Mark return as processed
     */
    public function processedEvent()
    {
        Event::dispatch(new ReturnProcessed($this));
    }

    /**
     * Cancel a return
     */
    public function cancelEvent()
    {
        Event::dispatch(new ReturnCancelled($this));
    }

    /**
     * Check if return request have items that are marked to be refunded
     *
     * @return bool
     */
    public function hasRefundItems()
    {
        return $this->items()
            ->whereHas('reason', function ($reason_query) {
                return $reason_query->where('type', 'refund');
            })->exists();
    }

    /**
     * Get Refunded total from the Return Request
     *
     * @return float
     */
    public function getRefundedTotalAttribute(): float
    {
        $refunded_amount = 0.00;
        if (config('returns.refund_amounts_enabled') !== true) {
            return $refunded_amount;
        }

        $is_completed = in_array($this->status_id, config('returns.completion_statuses'));

        if ($is_completed) {
            $refunded_amount = $this->items->sum('refunded_amount'); // refunded amount of items
            $refunded_amount += $this->delivery_refund_amount; // refunded delivery amount

            $refunded_amount = round($refunded_amount, 2);
        }

        return $refunded_amount;
    }
}
