// Debug function to log information
function debugRevolutPay(message, data = null) {
    const debugInfo = document.getElementById('revolut-debug-info');
    const timestamp = new Date().toISOString();
    let logMessage = `[${timestamp}] ${message}`;
    if (data) {
        logMessage += '\n' + JSON.stringify(data, null, 2);
    }
    if (debugInfo) {
        debugInfo.textContent += logMessage + '\n\n';
    }
    console.log(message, data);
}

// Initialize Revolut Pay
function initializeRevolutPay(config) {
    console.log('Initializing Revolut Pay with config:', config);
    
    // Validate configuration
    if (!config.public_key) {
        console.error('Missing public key in configuration');
        return;
    }

    if (!config.create_order_url) {
        console.error('Missing create order URL in configuration');
        return;
    }

    // Get the container
    const container = document.getElementById('revolut-pay-container');
    if (!container) {
        console.error('Revolut Pay container not found');
        return;
    }

    console.log('Found container:', container);
    console.log('Container data attributes:', {
        publicKey: container.dataset.publicKey,
        testMode: container.dataset.testMode,
        createOrderUrl: container.dataset.createOrderUrl,
        amount: container.dataset.amount,
        currency: container.dataset.currency,
        email: container.dataset.email
    });

    // Initialize Revolut Pay
    console.log('Initializing Revolut Pay with SDK:', window.RevolutCheckout);
    
    // Create the payment button
    const buttonContainer = document.getElementById('revolut-pay-button');
    if (!buttonContainer) {
        console.error('Button container not found');
        return;
    }
    console.log('Found button container:', buttonContainer);

    // Initialize the payment button
    const revolutPay = window.RevolutCheckout.revolutPay({
        publicKey: config.public_key,
        mode: config.test_mode ? 'test' : 'live',
        onSuccess: () => {
            console.log('Payment successful');
            // Handle successful payment
            window.location.href = config.success_url;
        },
        onError: (error) => {
            console.error('Payment failed:', error);
            // Handle payment error
            const errorMessage = document.getElementById('revolut-error-message');
            if (errorMessage) {
                errorMessage.textContent = 'Payment failed: ' + error.message;
                errorMessage.style.display = 'block';
            }
        }
    });

    // Create and mount the button
    const button = document.createElement('button');
    button.id = 'revolut-pay-button-element';
    button.className = 'revolut-pay-button';
    button.textContent = 'Pay with Revolut';
    button.style.cssText = 'width: 100%; padding: 10px; background-color: #000; color: #fff; border: none; border-radius: 4px; cursor: pointer;';
    
    buttonContainer.appendChild(button);

    // Add click handler
    button.addEventListener('click', async (e) => {
        e.preventDefault();
        console.log('Pay button clicked');

        try {
            // Show loading state
            const loadingElement = document.getElementById('revolut-pay-loading');
            if (loadingElement) {
                loadingElement.style.display = 'block';
            }

            // Create order
            console.log('Creating order with:', {
                amount: config.amount,
                currency: config.currency,
                email: config.email
            });

            const response = await fetch(config.create_order_url, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content
                },
                body: JSON.stringify({
                    amount: config.amount,
                    currency: config.currency,
                    email: config.email
                })
            });

            if (!response.ok) {
                const errorText = await response.text();
                console.error('Server error:', errorText);
                throw new Error('Failed to create order: ' + (response.status === 500 ? 'Internal server error' : errorText));
            }

            const order = await response.json();
            console.log('Order created:', order);

            // Show the payment popup
            revolutPay.show();
        } catch (error) {
            console.error('Failed to create order:', error);
            const errorMessage = document.getElementById('revolut-error-message');
            if (errorMessage) {
                errorMessage.textContent = 'Failed to initialize payment: ' + error.message;
                errorMessage.style.display = 'block';
            }
        } finally {
            // Hide loading state
            const loadingElement = document.getElementById('revolut-pay-loading');
            if (loadingElement) {
                loadingElement.style.display = 'none';
            }
        }
    });
}

// Load Revolut Pay script
window.loadRevolutPayScript = function() {
    console.log('Loading Revolut Pay script...');
    
    // Get configuration from the container
    const container = document.getElementById('revolut-pay-container');
    if (!container) {
        console.error('Revolut Pay container not found');
        return;
    }

    console.log('Found container with data attributes:', container.dataset);

    const config = {
        public_key: container.dataset.publicKey,
        test_mode: container.dataset.testMode === 'true',
        create_order_url: container.dataset.createOrderUrl,
        success_url: container.dataset.successUrl,
        amount: parseFloat(container.dataset.amount),
        currency: container.dataset.currency,
        email: container.dataset.email
    };

    console.log('Parsed config:', config);

    // Initialize Revolut Pay
    initializeRevolutPay(config);
};
