<?php

namespace Mtcmedia\RevolutPaymentProvider\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class RevolutPaymentService
{
    protected $apiKey;
    protected $baseUrl;
    protected $isTestMode;

    public function __construct()
    {
        $this->apiKey = config('revolut.api_key');
        $this->baseUrl = config('revolut.base_url');
        $this->isTestMode = config('revolut.test_mode', false);
    }

    public function createOrder($amount, $currency)
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json'
            ])->post($this->baseUrl . '/orders', [
                'amount' => $amount,
                'currency' => $currency,
                'capture_mode' => 'AUTOMATIC',
                'merchant_order_ext_ref' => uniqid('order_'),
                'email' => request()->user()->email ?? null,
            ]);

            if ($response->successful()) {
                return $response->json();
            }

            Log::error('Revolut order creation failed', [
                'status' => $response->status(),
                'response' => $response->json()
            ]);

            throw new \Exception('Failed to create Revolut order');
        } catch (\Exception $e) {
            Log::error('Revolut order creation error', [
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    public function getOrder($orderId)
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json'
            ])->get($this->baseUrl . '/orders/' . $orderId);

            if ($response->successful()) {
                return $response->json();
            }

            throw new \Exception('Failed to get Revolut order');
        } catch (\Exception $e) {
            Log::error('Revolut get order error', [
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    public function handleWebhook($payload)
    {
        try {
            $eventType = $payload['event'] ?? null;
            $orderId = $payload['order_id'] ?? null;

            if (!$eventType || !$orderId) {
                throw new \Exception('Invalid webhook payload');
            }

            $order = $this->getOrder($orderId);

            switch ($eventType) {
                case 'ORDER_PAYMENT_COMPLETED':
                    return $this->handlePaymentCompleted($order);
                case 'ORDER_PAYMENT_FAILED':
                    return $this->handlePaymentFailed($order);
                default:
                    Log::info('Unhandled Revolut webhook event', [
                        'event' => $eventType,
                        'order_id' => $orderId
                    ]);
                    return true;
            }
        } catch (\Exception $e) {
            Log::error('Revolut webhook handling error', [
                'error' => $e->getMessage(),
                'payload' => $payload
            ]);
            throw $e;
        }
    }

    protected function handlePaymentCompleted($order)
    {
        // Implement your payment completion logic here
        // For example, update order status, send confirmation email, etc.
        return true;
    }

    protected function handlePaymentFailed($order)
    {
        // Implement your payment failure logic here
        // For example, update order status, notify customer, etc.
        return true;
    }
}
