<?php

namespace Mtc\RevolutPaymentProvider;

use Illuminate\Support\ServiceProvider;
use Mtc\RevolutPaymentProvider\Classes\RevolutPay;
use Mtc\RevolutPaymentProvider\Classes\Operations\Refund;
use Mtc\RevolutPaymentProvider\Controllers\RevolutOrderController;
use Mtc\RevolutPaymentProvider\Services\RevolutService;
use Mtc\RevolutPaymentProvider\Middleware\RevolutCorsMiddleware;
use Mtc\RevolutPaymentProvider\Console\Commands\CreateOrderWebhook;
use Illuminate\Support\Facades\Event;


class RevolutServiceProvider extends ServiceProvider
{
    /**
     * Register services.
     */
    public function register()
    {
        // Merge configuration
        $this->mergeConfigFrom(
            __DIR__.'/../config/revolut.php', 'revolut'
        );

        // Register the CORS middleware
        $this->app['router']->aliasMiddleware('revolut.cors', RevolutCorsMiddleware::class);
    }

    /**
     * Bootstrap services.
     */
    public function boot()
    {
        // Publish configuration
        $this->publishes([
            __DIR__.'/../config/revolut.php' => config_path('revolut.php'),
        ], 'config');

        $this->publishes([
            __DIR__.'/../database/migrations/' => database_path('migrations')
        ],"database");

        $this->publishes([
            __DIR__.'/../database/seeders/' => database_path('seeders')
        ],"settings-seeders");


        // Publish assets
        $this->publishes([
            __DIR__.'/../resources/js' => public_path('js/revolut'),
            __DIR__.'/../resources/css' => public_path('css/revolut'),
        ], 'assets');

        // Load routes
        $this->loadRoutesFrom(__DIR__.'/../routes/revolutapi.php');
        $this->loadRoutesFrom(__DIR__.'/../routes/revolut-web.php');

        // Add the templates directory to the view paths
        $view_paths = config('view.paths', []);
        $view_paths[] = __DIR__.'/../resources/templates';
        config(['view.paths' => $view_paths]);

        // Register the template namespace
        $this->app['view']->addNamespace('revolut', __DIR__.'/../resources/templates');

        // Share configuration with views
        $this->app['view']->share('revolut_config', config('revolut'));


        // Listen for payment gateway loading
        $this->app['events']->listen('shop.checkout.load_payment_gateways', function ($basket) {
            $revolutPay = new RevolutPay(config('revolut'));
            return $revolutPay->form($basket);
        });

        $this->app['events']->listen('refunds.get_refund_methods', function () {
            return [
                'id' => 'revolut',
                'name' => 'Revolut Refund',
                'callback' => '\\' . Refund::class . '::handleRefund'
            ];
        });


        // Register console commands only when running in the console
        if ($this->app->runningInConsole()) {
            $this->commands([
                CreateOrderWebhook::class,
            ]);
        }
    }


}
