<?php

namespace Mtc\RevolutPaymentProvider;

use Illuminate\Support\ServiceProvider;
use Mtc\RevolutPaymentProvider\Classes\RevolutPay;
use Mtc\RevolutPaymentProvider\Classes\Operations\Refund;
use Mtc\RevolutPaymentProvider\Controllers\RevolutOrderController;
use Mtc\RevolutPaymentProvider\Services\RevolutService;
use Mtc\RevolutPaymentProvider\Middleware\RevolutCorsMiddleware;
use Mtc\RevolutPaymentProvider\Console\Commands\CreateOrderWebhook;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;


class RevolutServiceProvider extends ServiceProvider
{
    /**
     * Register services.
     */
    public function register()
    {
        // Merge configuration
        $this->mergeConfigFrom(
            __DIR__.'/../config/revolut.php', 'revolut'
        );
        $this->app['events']->listen('refunds.get_refund_methods', function () {
            return [
                'id' => 'revolut',
                'name' => 'Online Revolut Refund',
                'callback' => '\\' . Refund::class . '::handleRefund'
            ];
        });

        // Register the CORS middleware
        $this->app['router']->aliasMiddleware('revolut.cors', RevolutCorsMiddleware::class);
    }

    protected function bootSettings()
    {
        // Check if the 'settings' table exists
        if (Schema::hasTable('settings')) {
            // Get current revolut config values
            $currentConfig = config('revolut');

            // List the keys you'd want to check
            $requiredKeys = [
                'api_key',
                'api_public_key',
                'api_url',
                'webhook.secret',
                'webhook.id',
                'test_mode',
                'revolut_enabled',
            ];

            // Determine if any required config key is missing or empty
            $needsOverride = false;
            foreach ($requiredKeys as $key) {
                if (blank(data_get($currentConfig, $key))) {
                    $needsOverride = true;
                    break;
                }
            }

            if ($needsOverride) {
                // Pull values from the DB
                $settings = DB::table('settings')
                    ->where('module', 'revolut')
                    ->pluck('value', 'key')
                    ->toArray();

                if(!empty($settings)) {
                    // Override only if values are missing
                    config([
                        'revolut.api_key' => data_get($currentConfig, 'api_key') ?: (function() use ($settings) {
                            if($settings['REVOLUT_TEST_MODE']){
                                return $settings['REVOLUT_SANDBOX_API_KEY'];
                            }
                            return $settings['REVOLUT_LIVE_API_KEY'];
                        })(),
                        'revolut.revolut_enabled' => data_get($currentConfig, 'revolut_enabled') ?: (function() use ($settings) {
                            return $settings['REVOLUT_ENABLED'];
                        })(),
                        'revolut.api_public_key' => data_get($currentConfig, 'api_key') ?: (function() use ($settings) {
                            if($settings['REVOLUT_TEST_MODE']){
                                return $settings['REVOLUT_SANDBOX_PUBLIC_KEY'];
                            }
                            return $settings['REVOLUT_LIVE_PUBLIC_KEY'];
                        })(),
                        'revolut.api_url' => data_get($currentConfig, 'api_url') ?: (function() use ($settings) {
                            if($settings['REVOLUT_TEST_MODE']){
                                return $settings['REVOLUT_SANDBOX_API_URL'];
                            }
                            return $settings['REVOLUT_LIVE_API_URL'];
                        })(),
                        'revolut.webhook.secret' => data_get($currentConfig, 'webhook.secret') ?: (function() use ($settings) {
                            if($settings['REVOLUT_TEST_MODE']){
                                return $settings['REVOLUT_SANDBOX_WEBHOOK_SECRET'];
                            }
                            return $settings['REVOLUT_LIVE_WEBHOOK_SECRET'];
                        })(),
                        'revolut.webhook.id' => data_get($currentConfig, 'webhook.id') ?: (function() use ($settings) {
                            if($settings['REVOLUT_TEST_MODE']){
                                return $settings['REVOLUT_SANDBOX_WEBHOOK_ID'];
                            }
                            return $settings['REVOLUT_LIVE_WEBHOOK_ID'];
                        })(),
                    ]);
                }


            }
        }
    }

    /**
     * Bootstrap services.
     */
    public function boot()
    {

        $this->bootSettings();

        // Publish configuration
        $this->publishes([
            __DIR__.'/../config/revolut.php' => config_path('revolut.php'),
        ], 'config');

        $this->publishes([
            __DIR__.'/../database/migrations/' => database_path('migrations'),
            __DIR__.'/../database/seeders/' => database_path('seeders')

        ],"database");


        // Publish assets
        $this->publishes([
            __DIR__.'/../resources/js' => public_path('js/revolut'),
            __DIR__.'/../resources/css' => public_path('css/revolut'),
        ], 'assets');

        // Load routes
        $this->loadRoutesFrom(__DIR__.'/../routes/revolutapi.php');
        $this->loadRoutesFrom(__DIR__.'/../routes/revolut-web.php');

        // Add the templates directory to the view paths
        $view_paths = config('view.paths', []);
        $view_paths[] = __DIR__.'/../resources/templates';
        config(['view.paths' => $view_paths]);

        // Register the template namespace
        $this->app['view']->addNamespace('revolut', __DIR__.'/../resources/templates');

        // Share configuration with views
        $this->app['view']->share('revolut_config', config('revolut'));


        // Listen for payment gateway loading
        $this->app['events']->listen('shop.checkout.load_payment_gateways', function ($basket) {
            $revolutPay = new RevolutPay(config('revolut'));
            return $revolutPay->form($basket);
        });



        // Register console commands only when running in the console
        if ($this->app->runningInConsole()) {
            $this->commands([
                CreateOrderWebhook::class,
            ]);
        }
    }


}
