<?php

namespace Mtc\Sagepay;

use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\PayableContract;
use Mtc\Checkout\Contracts\PaymentGateway;
use Mtc\Checkout\PaymentForm;

/**
 * Sagepay Payment Gateway
 *
 * @package  Mtc\Sagepay
 */
class Sagepay implements PaymentGateway
{
    /**
     * Check if the gateway is available for use on this payment.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return bool
     */
    public function isApplicable(InvoiceRepositoryContract $invoice, $payable): bool
    {
        if ($invoice->getOutstandingAmount() <= 0.01) {
            return false;
        }

        return App::make(config('sagepay.applicable_check_class'))->handle($invoice, $payable);
    }

    /**
     * Render the payment template.
     *
     * @param InvoiceRepositoryContract $invoice
     * @return PaymentForm
     */
    public function getPaymentForm(InvoiceRepositoryContract $invoice): PaymentForm
    {

        // Get the vendor and create a unique vendor code for Sagepay.
        $vendor = config('sagepay.vendor_name');
        $vendor_code = $vendor . '-' . date('YmdHi') . '-' . $invoice->getId();

        $billing = $shipping = $invoice->getModel()->recipient;

        // Build up the encrypted form.
        $crypt = [
            'VendorTxCode' => $vendor_code,
            'Amount' => sprintf("%.2f", $invoice->getOutstandingAmount()),
            'Currency' => $invoice->getCurrency(),
            'Description' => config('app.name') . ' Merchandise',
            'SuccessURL' => route('charge_payment', [$invoice->getId(), 'gateway' => 'sagepay']),
            'FailureURL' => route('failed_payment'),
            'CustomerName' => $this->stripQuotes($billing->first_name . ' ' . $billing->last_name),
            'CustomerEMail' => $invoice->getEmail(),
            'VendorEMail' => config('sagepay.vendor_email'),
            'SendEMail' => config('sagepay.send_email'),
            'EmailMessage' => config('sagepay.email_message'),
            'BillingSurname' => $this->stripQuotes($billing->last_name),
            'BillingFirstnames' => $this->stripQuotes($billing->first_name),
            'BillingAddress1' => $this->stripQuotes($billing->address1),
            'BillingCity' => $this->stripQuotes($billing->city ?? ''),
            'BillingPostCode' => $billing->postcode ?? '',
            'BillingCountry' => $billing->country,
            'BillingPhone' => $invoice->getModel()->contact_number,
            'DeliverySurname' => $this->stripQuotes($shipping->last_name),
            'DeliveryFirstnames' => $this->stripQuotes($shipping->first_name),
            'DeliveryAddress1' => $this->stripQuotes($shipping->address1),
            'DeliveryCity' => $this->stripQuotes($shipping->city ?? ''),
            'DeliveryPostCode' => $shipping->postcode ?? '',
            'DeliveryCountry' => $shipping->country,
            'DeliveryPhone' => $invoice->getModel()->contact_number,
            'ReferrerID' => '968FB4B0-E929-45E4-A5BC-41A4FFE873F2',
        ];

        if (!empty($billing->address2)) {
            $crypt['BillingAddress2'] = $billing->address2;
        }

        if (!empty($shipping->address2)) {
            $crypt['DeliveryAddress2'] = $shipping->address2;
        }

        if (!empty($billing->state)) {
            $crypt['BillingState'] = $billing->state;
        }

        if (!empty($shipping->state)) {
            $crypt['DeliveryState'] = $shipping->state;
        }

        // Build the encryption string. Sagepay does not like http_build_query()!
        $post_values = collect($crypt)
            ->map(function ($value, $key) {
                return "{$key}=" . trim($value);
            })->implode('&');

        $template = template('sagepay/payment_form.twig', [
            'sagepay_live' => config('sagepay.use_production_endpoint'),
            'action' => $this->endpoint(config('sagepay.use_production_endpoint')),
            'form_fields' => [
                'VPSProtocol' => '3.00',
                'TxType' => 'PAYMENT',
                'Vendor' => config('sagepay.vendor_name'),
                'Crypt' => $this->encryptAndEncode($post_values),
            ],
        ]);

        return new PaymentForm($template, 'template', [
            'name' => __('sagepay::sagepay.payment_option_name'),
        ]);
    }

    /**
     * Charge payment on invoice
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return bool
     * @throws Exception
     */
    public function charge(Request $request, InvoiceRepositoryContract $invoice): array
    {
        $response = $this->decode(str_replace(' ', '+', $request->input('crypt')));
        if (!in_array($response['Status'], config('sagepay.successful_payment_statuses', []))) {
            throw new Exception("Sagepay Form transaction failed. The Error Message was " . $response['StatusDetail']);
        }

        return [
            'provider' => 'sagepay',
            'amount' => $response['Amount'],
            'currency_code' => $invoice->getCurrency(),
            'amount_in_currency' => $invoice->getOutstandingAmountInCurrency(),
            'reference' => $response['VPSTxId'],
            'details' => $response,
            'confirmed_at' => now(),
        ];
    }

    /**
     * Get the endpoint URL based on environment
     *
     * @param $live_mode
     * @return string
     */
    protected function endpoint($live_mode)
    {
        return $live_mode
            ? 'https://live.sagepay.com/gateway/service/vspform-register.vsp'
            : 'https://test.sagepay.com/gateway/service/vspform-register.vsp';
    }

    /**
     * Encrypt and encode in a way Sagepay will be able to decrypt.
     *
     * @param string $input Unencrypted query string.
     * @return string
     */
    protected function encryptAndEncode($input)
    {
        $input = $this->pkcs5Pad($input, 16);
        $password = config('sagepay.encryption_password');
        return "@" . bin2hex(openssl_encrypt($input, 'AES-128-CBC', $password, OPENSSL_RAW_DATA, $password));
    }

    /**
     * Take Sagepay encrypted query string and decode this into a format
     * we can read.
     *
     * @param string $input Encrypted query string.
     * @return string
     */
    protected function decodeAndDecrypt($input)
    {
        $input = substr($input, 1);
        $input = pack('H*', $input);
        $password = config('sagepay.encryption_password');
        return openssl_decrypt($input, 'AES-128-CBC', $password, OPENSSL_RAW_DATA, $password);
    }

    /**
     * Add padding as required for openssl_decrypt
     *
     * @param string $text Incoming text to pad.
     * @param int $block_size Number of blocks to pad by.
     * @return string
     */
    protected function pkcs5Pad($text, $block_size)
    {
        $padding = "";

        // Pad input to an even block size boundary
        $pad_length = $block_size - (strlen($text) % $block_size);
        for ($i = 1; $i <= $pad_length; $i++) {
            $padding .= chr($pad_length);
        }

        return $text . $padding;
    }

    /**
     * Decode the incoming encrypted query string and return the result
     * as an unencrypted array of variables.
     *
     * @param string $strIn Encrypted query string.
     * @return array
     */
    protected function decode($strIn)
    {
        $decoded_string = $this->decodeAndDecrypt($strIn);
        // Convert the decoded query string into $sagepay_response.
        parse_str($decoded_string, $sagepay_response);
        return $sagepay_response;
    }

    protected function stripQuotes($input): string
    {
        return str_replace([ '"', "'", '’', '`'], '', $input);
    }
}
