<?php

namespace Mtc\ShippingManager\DeliveryTypes;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Mtc\Basket\Contracts\BasketRepositoryInterface;
use Mtc\ShippingManager\CollectionPoint;
use Mtc\ShippingManager\Contracts\DeliveryMethodContract;

/**
 * Class TableRateDelivery
 *
 * @package Mtc\ShippingManager
 */
class CollectionDelivery implements DeliveryMethodContract
{
    /**
     * Retrieve methods that are applicable to basket
     *
     * @param BasketRepositoryInterface $basket
     */
    public function methodsForBasket(BasketRepositoryInterface $basket)
    {
        $locations = Cache::rememberForever('collection-locations', function () {
            return CollectionPoint::query()
                ->get();
        });

        $this->setActiveMethod($basket, $locations);
        return $locations;
    }

    /**
     * Set active delivery method
     *
     * @param BasketRepositoryInterface $basket
     * @param Collection $rates
     */
    public function setActiveMethod(BasketRepositoryInterface $basket, Collection $locations)
    {
        if ($this->hasInvalidCollectionSurcharge($basket, $locations)) {
            if ($locations->count() == 0) {
                $basket->removeSurchargeByType('collection');
                return;
            }

            $basket->updateSurchargeByType('collection', $locations->first()->id);
        }

        $this->setActiveLocationInList($basket, $locations);
    }

    /**
     * Check if the delivery surcharge applied to basket matches the location list
     *
     * @param BasketRepositoryInterface $basket
     * @param Collection $locations
     * @return bool
     */
    protected function hasInvalidCollectionSurcharge(BasketRepositoryInterface $basket, Collection $locations)
    {
        if (!$basket->hasSurcharge(0, 'collection')) {
            return true;
        }

        return $locations->where('id', $basket->getSurchargeIdFromType('collection'))->count() == 0;
    }

    /**
     * Check and set the active location as active in the list based on basket
     *
     * @param BasketRepositoryInterface $basket
     * @param $locations
     */
    protected function setActiveLocationInList(BasketRepositoryInterface $basket, $locations)
    {
        $locations->each(function ($location) use ($basket) {
            if ($basket->hasSurcharge($location->id, 'collection')) {
                $location->active = true;
            }
        });
    }
}
