<?php

namespace Mtc\ShippingManager\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\Foundation\Country;
use Mtc\Foundation\Http\Resources\CountryList;
use Mtc\ShippingManager\CollectionPoint;
use Mtc\ShippingManager\Http\Requests\CollectionPointRequest;
use Mtc\ShippingManager\ShippingMethod;
use Mtc\ShippingManager\Zone;

/**
 * Class ShippingController
 *
 * @package Mtc\ShippingManager
 */
class CollectionPointController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return string
     */
    public function index()
    {
        $this->page_meta['title'] = $this->page_meta['page_name'] = 'Manage Collection Points';
        return template('/admin/collection_points/index.twig', [
            'page_meta' => $this->page_meta,
            'collect_locations' => CollectionPoint::all(),
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return string
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Syntax
     */
    public function create()
    {
        $this->page_meta['title'] = $this->page_meta['page_name'] = 'Add Collection Points';
        return template('/admin/collection_points/edit.twig', [
            'page_meta' => $this->page_meta,
            'countries' => Country::query()->pluck('name', 'code'),
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param Request $request
     * @return mixed
     */
    public function store(CollectionPointRequest $request)
    {
        $location = CollectionPoint::query()
            ->create($request->input());

        session()->flash('success', 'Shipping Method Created');
        return redirect()->to(route('admin.collection-points.edit', [ $location->id ]));
    }


    /**
     * Show the form for editing the specified resource.
     *
     * @param CollectionPoint $location
     * @return string
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Syntax
     */
    public function edit(CollectionPoint $collection_point)
    {
        $this->page_meta['title'] = $this->page_meta['page_name'] = 'Edit Collection Point';
        return template('/admin/collection_points/edit.twig', [
            'page_meta' => $this->page_meta,
            'collection_point' => $collection_point,
            'countries' => Country::query()->pluck('name', 'code'),
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param Request $request
     * @param $collect_location_id
     * @return mixed
     */
    public function update(CollectionPointRequest $request, $shipping_zone_id)
    {
        $this->validate($request, [
            'name' => 'required',
            'type' => 'required',
            'parent_id' => 'required|numeric'
        ]);

        $shipping_zone = Zone::query()->findOrFail($shipping_zone_id);
        if ($request->has('postal_zones')) {
            $shipping_zone->postal_zones = $request->input('postal_zones');
        }

        $shipping_zone->fill($request->all())->save();

        if ($request->has('countries')) {
            $shipping_zone->countries()->sync($request->input('countries'));
        }


        session()->flash('success', 'Shipping Zone updated');
        return redirect()->back();
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param $zone_id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy($zone_id)
    {
        $zone = Zone::query()->findOrFail($zone_id);
        $zone->delete();

        session()->flash('success', 'Zone deleted');
        return redirect()->back();
    }

}
