<?php

namespace Mtc\ShippingManager\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Mtc\ShippingManager\Jobs\DispatchShipmentToCourier;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\Orders\Order;
use Mtc\Orders\OrderShipment;

/**
 * Class ExportShipmentController
 *
 * @package Mtc\ShippingManager
 */
class ExportShipmentController extends Controller
{
    /**
     * Handle shipment export to courier
     *
     * @param Request $request
     * @param Order $order
     * @param OrderShipment $shipment
     * @return \Illuminate\Http\RedirectResponse
     * @throws \Illuminate\Validation\ValidationException
     */
    public function __invoke(Request $request, Order $order, OrderShipment $shipment)
    {
        if ($request->has('box_count')) {
            $this->processRequest($request, $shipment);
        }
        $this->dispatch(new DispatchShipmentToCourier($shipment));
        session()->flash('messages', [
            'type' => 'success',
            'content' => 'Shipment Sent to Courier'
        ]);
        return redirect()->back();
    }

    /**
     * Basic mode - user sets number of boxes without building them up individually
     *
     * @param Request $request
     * @param OrderShipment $shipment
     * @throws \Illuminate\Validation\ValidationException
     */
    protected function processRequest(Request $request, OrderShipment $shipment)
    {
        $validated = $this->validate($request, [
            'box_count' => 'required|numeric|min:1',
            'weight' => 'required|numeric|min:0.01',
            'shipment_value' => 'required|numeric|min:0.01',
            'service_id' => 'required|numeric',
        ]);

        $shipment->fill($validated)->save();
        $shipment->boxes()->delete();

        $boxes = range(1, $validated['box_count']);
        $weight_per_box =  $validated['weight'] / $validated['box_count'];
        foreach ($boxes as $box) {
            $shipment->boxes()
                ->create([
                    'weight' => $weight_per_box
                ]);
        }
    }
}
