<?php

namespace Mtc\ShippingManager;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Mtc\Core\UkPostcode;
use Mtc\Money\HasPrices;
use Mtc\ShippingManager\Scopes\OrderedRate;

/**
 * Class ShippingZoneRate
 *
 * @author  Haroldas Latonas
 */
class TableRate extends Model
{
    use HasPrices;

    /**
     * constants for price & weight names
     */
    public const PRICE_RATE = 'price';
    public const WEIGHT_RATE = 'weight';

    /**
     * Rate types
     *
     * @var array
     */
    public static $rate_types = [
        'price' => 'Price (£)',
        'weight' => 'Weight (kg)',
    ];

    /**
     * @var string $table
     */
    protected $table = 'shipping_zone_table_rates';

    /**
     * @var array $fillable
     */
    protected $fillable = [
        'zone_id',
        'name',
        'description',
        'type',
        'rate_type',
        'range_min',
        'range_max',
        'rate',
        'order',
        'delivery_courier_id',
        'delivery_service_id',
        'modifiers',
    ];

    /**
     * @var array $casts
     */
    protected $casts = [
        'disabled' => 'boolean',
        'modifiers' => 'array'
    ];

    /**
     * Set fields as visible when returning json
     *
     * @var array
     */
    protected $visible = [
        'id',
        'name',
        'rate',
        'active',
        'display_price',
        'description',
    ];

    /**
     * Append variable to toArray method
     *
     * @var array
     */
    protected $appends = [
        'display_price'
    ];

    /**
     * Append variable to toArray method
     *
     * @var array
     */
    protected $price_fields = [
        'rate'
    ];

    /**
     * Model booting
     */
    protected static function boot()
    {
        parent::boot();
        self::addGlobalScope(new OrderedRate());
    }

    /**
     * Search rates that apply to address / weight / price combo
     *
     * @param $address
     * @param $weight
     * @param $price
     * @return Collection
     */
    public static function search($address, $weight, $price)
    {
        $zones = Zone::zonesForCountry($address->country);
        $zones = self::filterPostalCodeZones($address, $zones);

        if ($zones->isEmpty()) {
            return collect([]);
        }

        return self::query()
            ->active()
            ->where(function (Builder $query) use ($zones) {
                return $query->whereIn('zone_id', $zones->pluck('id'))
                    ->orWhere('zone_id', 0);
            })
            ->withinRange('weight', $weight)
            ->withinRange('price', $price)
            ->get()
            ->sortBy(function ($rate) {
                if (in_array('admin-user', $rate->modifiers ?? [])) {
                    return 99999;
                }

                return $rate->rate->raw();
            });
    }

    /**
     * Filter out postcode zones
     *
     * @param $address
     * @param $zones
     * @return mixed
     */
    protected static function filterPostalCodeZones($address, Collection $zones)
    {
        $postcode_zones = $zones
            ->map(function ($zone) use ($address) {
                return $zone->subZones
                    ->filter(function (Zone $sub_zone) use ($address) {
                        if ($sub_zone->type === 'postal') {
                            $zone_data = json_decode($sub_zone->postal_zones);
                            return self::checkZone($zone_data, $address->postcode);
                        }
                        return false;
                    });
            })
            ->flatten();

        if ($postcode_zones->isNotEmpty()) {
            $zones = $zones->reject(function ($zone) use ($postcode_zones) {
                return in_array($zone->id, $postcode_zones->pluck('parent_id')->toArray());
            });
            $zones = $zones->merge($postcode_zones);
        }
        return $zones;
    }
    /**
     * Only active rates
     *
     * @param Builder $query
     * @return Builder
     */
    public function scopeActive(Builder $query)
    {
        return $query->where('disabled', 0);
    }

    /**
     * Filter to specific range
     *
     * @param Builder $query
     * @param $type
     * @param $value
     * @return Builder
     */
    public function scopeWithinRange(Builder $query, $type, $value)
    {
        return $query->where('rate_type', '!=', $type)
            ->orWhere(function ($weight_query) use ($value, $type) {
                return $weight_query->where('rate_type', $type)
                    ->where('range_min', '<=', $value)
                    ->where('range_max', '>=', $value);
            });
    }

    /**
     * Relationship to zone that has this rate
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function zone()
    {
        return $this->belongsTo(Zone::class);
    }

    /**
     * Check if postcode is in specific zones
     *
     * @param $postcode_zones
     * @param $postcode
     */
    public static function checkZone($postcode_zones, $postcode): bool
    {
        $postcode = new UkPostcode($postcode);

        foreach ($postcode_zones as $postcode_zone) {
            // If the first part of the postcode matches a postcode zone
            // If the postcode zone has an additional restrictions [8..10]
            if ($postcode->getArea() === $postcode_zone->area && count($postcode_zone->ranges) > 0) {
                foreach ($postcode_zone->ranges as $range) {
                    if ($postcode->getDistrict() >= $range[0] && $postcode->getDistrict() <= $range[1]) {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    /**
     * Relationship with delivery courier
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function courier()
    {
        return $this->hasOne(DeliveryCourier::class, 'id', 'delivery_courier_id');
    }

    /**
     * Get name attribute
     *
     * @return mixed
     */
    public function getNameAttribute()
    {
        return $this->attributes['name'];
    }

    /**
     * @return mixed
     * @throws \Exception
     */
    public function getDisplayPriceAttribute()
    {
        return $this->fetchPriceAttribute('rate')->raw();
    }

    /**
     * @return mixed
     * @throws \Exception
     */
    public function getPriceAttribute()
    {
        return $this->fetchPriceAttribute('rate');
    }
}
