<?php

namespace Mtc\ShippingManager\Courier;

use GuzzleHttp\Client;
use Illuminate\Support\Facades\Log;

/**
 * Class CourierApi
 *
 * @package Mtc\ShippingManager
 */
class CourierApi
{
    /**
     * @var Client
     */
    protected $client;

    /**
     * CourierApi constructor.
     *
     * @param Client $client
     */
    public function __construct(Client $client)
    {
        $this->client = $client;
    }

    /**
     * Check if courier should interact with live environment
     *
     * @return bool
     */
    protected function isLive()
    {
        return config('app.env') === 'production'
            && config('courier.live');
    }

    /**
     * Get Delivery Service URL
     *
     * @param $endpoint
     */
    protected function apiLocation($endpoint): string
    {
        return $this->isLive()
            ? config('courier.delivery_service_url') . $endpoint
            : config('courier.delivery_service_test_url') . $endpoint;
    }

    /**
     * Get the appropriate auth token for the API
     *
     * @return mixed
     */
    protected function getToken()
    {
        return config('courier.api_token');
    }

    /**
     * Send the order to Courier
     *
     * @param array $shipment_data
     * @return bool
     * @throws \Exception
     */
    public function post($endpoint, $data = [], $return_status = false): bool
    {
        $data['api_token'] = $this->getToken();

        $client = new Client();
        $response = $client->post($this->apiLocation($endpoint), [
            'form_params' => $data,
        ]);

        if (config('app.debug')) {
            Log::info('POST to ' . $endpoint, [
                'request' => $data,
                'response' => json_decode((string)$response->getBody())
            ]);
        }
        return $return_status ? $response->getStatusCode() : json_decode((string)$response->getBody());
    }

    /**
     * Get available services for client
     *
     * @return mixed
     */
    public function get($endpoint)
    {
        $response = $this->client->post($this->apiLocation($endpoint), [
            'form_params' => [
                'api_token' => $this->getToken()
            ],
        ]);
        return json_decode((string)$response->getBody());
    }
}
