<?php

namespace Mtc\ShippingManager\Providers;

use Illuminate\Database\Eloquent\Relations\Relation;
use Mtc\ShippingManager\AdminUserShippingModifier;
use Mtc\ShippingManager\Console\Commands\InstallShippingManager;
use Illuminate\Support\Facades\App;
use Illuminate\Support\ServiceProvider;
use Mtc\ShippingManager\Contracts\DeliveryCourierOrderMap;
use Mtc\ShippingManager\Contracts\DeliveryMethodContract;
use Mtc\ShippingManager\Facades\ShippingModifiers;
use Mtc\ShippingManager\Shipping;
use Mtc\ShippingManager\ShippingModifierManager;

/**
 * Class ShippingServiceProvider
 * @package Mtc\Plugins\ShippingManager\Classes\Providers
 */
class ShippingServiceProvider extends ServiceProvider
{

    /**
     * Register plugin service provider
     */
    public function register()
    {
        // Load config
        $this->mergeConfigFrom(dirname(__DIR__, 2) . '/config/shipping_manager.php', 'shipping_manager');

        if (config('shipping_manager.enabled', false) === false) {
            return;
        }

        collect(config('shipping_manager.service_providers', []))
            ->each(function ($service_provider) {
                $this->app->register($service_provider);
            });

        $this->registerBindings();
    }

    /**
     * Boot plugin service provider
     */
    public function boot()
    {
        //Check if plugin enabled (Admin Settings)
        if (config('shipping_manager.enabled') !== true) {
            return;
        }

        $shipping_morph_map = collect(config("shipping_manager.shipping_types", []))
            ->map(function ($data, $type) {
                return $data['model'];
            })->toArray();

        Relation::morphMap($shipping_morph_map);

        //Discount functionality
        if (config('shipping_manager.discount') === true) {
            Relation::morphMap([
                config('shipping_manager.discount_relation_name') => App::make(config('shipping_manager.discount_class'))
            ]);
        }

        //Load translations
        $this->loadTranslationsFrom(dirname(__DIR__, 2) . '/resources/lang', 'shipping_manager');

        // Load migrations
        if ($this->app->runningInConsole()) {
            $this->loadMigrationsFrom(dirname(__DIR__, 2) . '/database/migrations/');
            $this->bootResourcePublish();

        } else {
            app('twig_loader')->addPath(dirname(__DIR__, 2) . '/resources/templates');
        }

        ShippingModifiers::register('admin-user', new AdminUserShippingModifier());
    }

    /**
     * Register bindings used by the package
     */
    protected function registerBindings()
    {
        App::bind(DeliveryCourierOrderMap::class, function () {
            return App::make(config('shipping_manager.order_map_to_courier'));
        });

        App::bind(DeliveryMethodContract::class, function () {
            return App::make(config('shipping_manager.shipping_types.delivery.class'));
        });

        App::bind('delivery_method_response_class', function () {
            return App::make(config('shipping_manager.delivery_method_response_class'));
        });

        App::bind('shipping', function () {
            return App::make(Shipping::class);
        });

        App::bind('delivery', function () {
            return App::make(DeliveryMethodContract::class);
        });

        // Register the main class to use with the facade
        $this->app->singleton('shipping-modifiers', function ($app) {
            return new ShippingModifierManager($app);
        });
    }

    /**
     * Register resources that get publishedx
     */
    protected function bootResourcePublish()
    {
        $this->publishes([
            dirname(__DIR__, 2) . '/config/' => config_path()
        ], 'config');

        $this->publishes([
            dirname(__DIR__, 2) . '/database/seeds/' => database_path('seeds')
        ], 'seeds');

        $this->publishes([
            dirname(__DIR__, 2) . '/resources/js/' => resource_path('js/vendor')
        ], 'assets');

        $this->publishes([
            dirname(__DIR__, 2) . '/resources/less/' => resource_path('less/vendor')
        ], 'assets');

    }
}
