<?php

namespace Mtc\ShippingManager\Courier;

use Illuminate\Support\Facades\File;
use Mpdf\Output\Destination;
use Mtc\Checkout\Invoice;
use Mtc\Orders\Order;
use Mtc\Orders\OrderShipment;
use Mtc\Orders\OrderShipmentItem;
use Mtc\ShippingManager\Contracts\CourierShipmentMapContract;

/**
 * Class ShipmentMap
 * Default integration for shipment mapping to courier
 *
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class ShipmentMap implements CourierShipmentMapContract
{
    /**
     * Mapping class needs to be able to map the order to the format required by API
     *
     * @param OrderShipment $shipment
     */
    public function map($shipment) : array
    {
        return [
            'id' => $shipment->id,
            'service_id' => $this->serviceId($shipment),
            'description' => __('shipping_manager::shipping_manager.export_item_name', [ 'site' => config('app.name')]),
            'order_total' => $this->shipmentValue($shipment),
            'items' => $this->mapItems($shipment->boxes, $shipment->shipment_value),
            'info' => $this->info($shipment),
            'invoice' => $this->invoiceData($shipment->order),
            'address' => [
                'billing' => $this->billingAddress($shipment),
                'shipping' => $this->shippingAddress($shipment)
            ],
        ];
    }

    /**
     * Get the service ID
     *
     * @param $shipment
     * @return mixed
     */
    protected function serviceId($shipment)
    {
        return $shipment->service_id;
    }

    /**
     * Get the shipment info fields
     *
     * @param $shipment
     * @return array
     */
    protected function info($shipment)
    {
        return [
            'email' => $shipment->order->email,
            'contact_no' => $shipment->order->contact_number,
        ];
    }

    /**
     * Get the shipment billing address fields
     *
     * @param $shipment
     * @return array
     */
    protected function billingAddress($shipment)
    {
        return [
            'organisation' => $shipment->order->billingAddress->first_name . ' ' . $shipment->order->billingAddress->last_name,
            'first_name' => $shipment->order->billingAddress->first_name,
            'last_name' => $shipment->order->billingAddress->last_name,
            'address1' => $shipment->order->billingAddress->address1,
            'address2' => $shipment->order->billingAddress->address2 ?? '',
            'city' => $shipment->order->billingAddress->city,
            'postcode' => $shipment->order->billingAddress->postcode,
            'state' => $shipment->order->billingAddress->state ?? '',
            'country' => $shipment->order->billingAddress->country,
        ];
    }

    /**
     * Get the shipment billing address fields
     *
     * @param $shipment
     * @return array
     */
    protected function shippingAddress($shipment)
    {
        return [
            'organisation' => $shipment->order->shippingAddress->first_name . ' ' . $shipment->order->shippingAddress->last_name,
            'first_name' => $shipment->order->shippingAddress->first_name,
            'last_name' => $shipment->order->shippingAddress->last_name,
            'address1' => $shipment->order->shippingAddress->address1,
            'address2' => $shipment->order->shippingAddress->address2 ?? '',
            'city' => $shipment->order->shippingAddress->city,
            'postcode' => $shipment->order->shippingAddress->postcode,
            'state' => $shipment->order->shippingAddress->state ?? '',
            'country' => $shipment->order->shippingAddress->country,
        ];
    }

    /**
     * Map order items for export to courier
     *
     * @param OrderShipmentItem[] $items
     */
    protected function mapItems($boxes, $shipment_value) : array
    {
        if ($boxes->isEmpty()) {
            return [];
        }
        $value_per_package = $shipment_value / $boxes->count();
        return collect($boxes)
            ->map(function ($box) use ($value_per_package) {
                return [
                    'name' => __('shipping_manager::shipping_manager.export_item_name', [ 'site' => config('app.name')]),
                    'quantity' => 1,
                    'price' => $value_per_package,
                    'weight' => $box->weight,
                ];
            })->toArray();
    }

    /**
     * Shipment value
     *
     * @param $shipment
     * @return mixed
     */
    protected function shipmentValue($shipment)
    {
        return $shipment->order->cost_total->withTax();
    }

    /**
     * Generate invoice for paperless trade
     *
     * @param Order $order
     * @return string|null
     */
    protected function invoiceData($shipment)
    {
        $invoice = Invoice::query()
            ->whereHas('order', function ($order_query) use ($shipment) {
                $order_query->where('id', $shipment->order_id);
            })
            ->with([
                'addresses',
                'items'
            ])
            ->first();

        $html = template(config('invoices.default_template'), [
            'invoice' => $invoice,
        ]);

        $mpdf = new \Mpdf\Mpdf();
        $stylesheet = File::get(public_path('css/print_order.css'));
        $mpdf->WriteHTML($stylesheet, \Mpdf\HTMLParserMode::HEADER_CSS);
        $mpdf->WriteHTML($html, \Mpdf\HTMLParserMode::HTML_BODY);

        return base64_encode($mpdf->Output('', Destination::STRING_RETURN));
    }
}
