<?php

namespace Mtc\ShippingManager;

use Illuminate\Support\Collection;
use Illuminate\Support\Manager;
use Mtc\Basket\Contracts\BasketRepositoryInterface;

/**
 * Class ShippingModifierManager
 *
 * @package Mtc\ShippingManager
 */
class ShippingModifierManager extends Manager
{
    /**
     * Get the default driver name.
     *
     * @return string
     */
    public function getDefaultDriver()
    {
        return $this->app['config']['shipping_manager.active_modifier'] ?? '';
    }

    /**
     * Set the default session driver name.
     *
     * @param  string  $name
     * @return void
     */
    public function setActiveDriver($name)
    {
        $this->app['config']['shipping_manager.active_modifier'] = $name;
    }

    /**
     * Register a new driver through the manager
     *
     * @param $name
     * @param $driver
     */
    public function register($name, $driver)
    {
        $this->app['shipping-modifiers']->extend($name, function() use ($driver) {
            return $driver;
        });
    }

    /**
     * Get the list of modifiers
     *
     * @return array
     */
    public function getAllModifiers()
    {
        $this->createMissingDrivers();
        return $this->drivers;
    }

    /**
     * Build up drivers that are not yet created
     */
    protected function createMissingDrivers()
    {
        collect(array_diff(array_keys($this->customCreators), array_keys($this->drivers)))
            ->each(function ($driver) {
                $this->driver($driver);
            });
    }

    /**
     * Apply modifiers to the rates
     *
     * @param BasketRepositoryInterface $basket
     * @param Collection $rates
     */
    public function apply(BasketRepositoryInterface $basket, Collection $rates)
    {
        $rate_modifier_data = $rates
            ->keyBy('id')
            ->map(function ($rate) use ($basket) {
                // No modifiers - no reason for rejection
                if (empty($rate->modifiers)) {
                    return [
                        'modifiers' => [],
                        'matched_modifiers' => [],
                        'mandatory_modifiers' => [],
                        'always_visible' => false,
                    ];
                }

                return $this->rateModifiers($rate, $basket);
            })
            ->filter(function ($modifiers) {
                return $this->missingMandatoryModifier($modifiers) === false
                    && $this->matchesOptionalModifiers($modifiers);
            });

        $matched_modifiers = $rate_modifier_data
            ->pluck('matched_modifiers')
            ->flatten()
            ->filter();

        $discard_modifiers = $rate_modifier_data
            ->pluck('discard_modifiers')
            ->flatten()
            ->filter(function ($modifier) use ($matched_modifiers) {
                return $modifier && $matched_modifiers->contains($modifier);
            })
            ->toArray();

        $rates_matched = $rate_modifier_data
            ->filter(function ($rate_modifiers) use ($discard_modifiers) {
                if ($rate_modifiers['always_visible']) {
                    return true;
                }
                return $this->hasDiscardModifiers($rate_modifiers, $discard_modifiers);
            })
            ->keys()
            ->toArray();

        return $rates->filter(function ($rate) use ($rates_matched) {
            return in_array($rate->id, $rates_matched);
        });
    }

    /**
     * Get the modifiers on rate
     * @param $rate
     * @param $basket
     * @return array
     */
    protected function rateModifiers($rate, $basket)
    {
        return [
            'modifiers' => $rate->modifiers,
            'matched_modifiers' => $this->rateMatchedModifiers($rate, $basket),
            'mandatory_modifiers' => $this->rateMandatoryModifiers($rate, $basket),
            'discard_modifiers' => $this->rateDiscardModifiers($rate, $basket),
            'always_visible' => $this->isRateVisibilityForced($rate, $basket),
        ];
    }

    /**
     * Get the modifiers that are matched for rate on basket
     *
     * @param $rate
     * @param $basket
     * @return array
     */
    protected function rateMatchedModifiers($rate, $basket)
    {
        return collect($rate->modifiers)
            ->map(function ($modifier) use ($basket, $rate) {
                // Avoid fatal error in case db has a modifier that does not exist
                if (array_key_exists($modifier, $this->customCreators) === false) {
                    return false;
                }
                $this->setActiveDriver($modifier);
                return $this->isApplicable($basket, $rate) ? $modifier : null;
            })
            ->filter()
            ->toArray();
    }

    /**
     * Get the modifiers that are mandatory for the rate to apply
     *
     * @param $rate
     * @param $basket
     * @return array
     */
    protected function rateMandatoryModifiers($rate, $basket)
    {
        return collect($rate->modifiers)
            ->map(function ($modifier) use ($basket, $rate) {
                // Avoid fatal error in case db has a modifier that does not exist
                if (array_key_exists($modifier, $this->customCreators) === false) {
                    return false;
                }
                $this->setActiveDriver($modifier);
                return $this->isMandatoryOnRate($basket, $rate) ? $modifier : null;
            })
            ->filter()
            ->toArray();
    }

    /**
     * Get the modifiers that are mandatory for the rate to apply
     *
     * @param $rate
     * @param $basket
     * @return boolean
     */
    protected function isRateVisibilityForced($rate, $basket)
    {
        return collect($rate->modifiers)
            ->filter(function ($modifier) use ($basket, $rate) {
                // Avoid fatal error in case db has a modifier that does not exist
                if (array_key_exists($modifier, $this->customCreators) === false) {
                    return false;
                }
                $this->setActiveDriver($modifier);
                return $this->isGloballyAvailable($basket, $rate);
            })
            ->isNotEmpty();
    }

    /**
     * Get the modifiers that will lock out other rates
     *
     * @param $rate
     * @param $basket
     * @return array
     */
    protected function rateDiscardModifiers($rate, $basket)
    {
        return collect($rate->modifiers)
            ->map(function ($modifier) use ($basket, $rate) {
                // Avoid fatal error in case db has a modifier that does not exist
                if (array_key_exists($modifier, $this->customCreators) === false) {
                    return false;
                }
                $this->setActiveDriver($modifier);
                return $this->willDiscardOtherRates($basket, $rate) ? $modifier : null;
            })
            ->filter()
            ->toArray();
    }

    /**
     * Check if there are modifiers that were not matched but ar mandatory
     *
     * @param $modifiers
     * @return bool
     */
    protected function missingMandatoryModifier($modifiers)
    {
        return !empty(array_diff($modifiers['mandatory_modifiers'], $modifiers['matched_modifiers']));
    }

    /**
     * Check if there are modifiers that were not matched but ar mandatory
     *
     * @param $modifiers
     * @return bool
     */
    protected function matchesOptionalModifiers($modifiers)
    {
        if (empty($modifiers['modifiers'])) {
            return true;
        }

        return !empty(array_intersect($modifiers['modifiers'], $modifiers['matched_modifiers']));
    }

    /**
     * Check if the discard modifiers are part of the rates modifiers
     *
     * @param $modifiers
     * @return bool
     */
    protected function hasDiscardModifiers($modifiers, $discard_modifiers = [])
    {
        if (empty($discard_modifiers)) {
            return true;
        }

        return empty(array_diff($discard_modifiers, $modifiers['matched_modifiers']));
    }
}
