<?php

namespace Mtc\ShippingManager\Courier;

use Carbon\Carbon;
use GuzzleHttp\Exception\ClientException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Mtc\Orders\OrderShipment;
use Mtc\ShippingManager\Contracts\CourierShipmentMapContract;
use Mtc\ShippingManager\Contracts\DeliveryCourierServiceContract;

/**
 * Class Courier
 * @package Mtc\Plugins\ShippingManager\Classes
 */
class CourierService implements DeliveryCourierServiceContract
{
    /**
     * @var CourierApi
     */
    protected $api;

    /**
     * CourierService constructor.
     *
     * @param CourierApi $api
     */
    public function __construct(CourierApi $api)
    {
        $this->api = $api;
    }

    /**
     * Send order to courier through api
     *
     * @param OrderShipment $shipment
     * @return bool
     */
    public function send($shipment): bool
    {
        try {
            $data = App::make(CourierShipmentMapContract::class)->map($shipment);
            return $this->api->post('orders', $data, true) === 201;
        } catch (ClientException $exception) {
            Log::error((string)$exception->getResponse()->getBody());
        } catch (\Exception $exception) {
            Log::error((string)$exception->getMessage());
        }
        return false;
    }

    /**
     * Get the services
     *
     * @return mixed
     */
    public function getServices(): array
    {
        try {
            return Cache::remember('courier-services', now()->addMinutes(15), function () {
                return collect($this->api->get('couriers'))
                    ->pluck('services')
                    ->flatten(1)
                    ->pluck('name', 'id')
                    ->toArray();
            });
        } catch (\Exception $exception) {
            return [];
        }
    }

    /**
     * Process incoming web-hook
     *
     * @param Request $request
     * @return bool whether request was accepted
     */
    public function processWebHook(Request $request): bool
    {
        if ($request->server('HTTP_AUTH') !== config('courier.api_token')) {
            return false;
        }

        return $this->isWebhookASuccessResponse($request)
            ? $this->saveShipmentInfo($request)
            : $this->saveErrorResponse($request);
    }

    /**
     * @param Request $request
     * @return bool
     */
    protected function isWebhookASuccessResponse(Request $request)
    {
        return $request->input('type') !== 'error';
    }

    /**
     * Save error information coming from webhook
     *
     * @param Request $request
     */
    protected function saveShipmentInfo(Request $request)
    {
        collect($request->input())
            ->each(function ($shipment) {
                OrderShipment::query()
                    ->updateOrCreate([
                        'id' => $shipment['order_id'],
                    ], [
                        'id' => $shipment['order_id'],
                        'courier_name' => $shipment['courier'] . ' - ' . $shipment['service'],
                        'exported_at' => Carbon::createFromFormat(Carbon::DEFAULT_TO_STRING_FORMAT, $shipment['created_at']),
                        'label' => $shipment['label'],
                        'label_format' => $shipment['label_format'],
                        'tracking_number' => $shipment['tracking_no'],
                        'shipment_confirmed_at' => Carbon::now(),
                        'shipment_data' => $shipment['payload'],
                    ]);
            });

        return true;
    }

    /**
     * Save error information coming from webhook
     *
     * @param Request $request
     */
    protected function saveErrorResponse(Request $request)
    {
        OrderShipment::query()
            ->updateOrCreate([
                'id' => $request->input('id')
            ], [
                'id' => $request->input('id'),
                'shipment_data' => $request->input()
            ]);

        return false;
    }
}
