<?php

namespace Mtc\ShippingManager;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;

/**
 * Zone model
 *
 * @author Haroldas Latonas
 */
class Zone extends Model
{
    /**
     * Default type for zone
     */
    public const DEFAULT_TYPE = 'country';

    /**
     * Supported types for zones
     */
    public const TYPES = [
        'country' => 'Country',
        'postal' => 'Postal',
    ];

    /**
     * Table of the model
     *
     * @var string $table
     */
    protected $table = 'shipping_zones';

    /**
     * Mass assignable attributes
     *
     * @var string[] $fillable
     */
    protected $fillable = [
        'name',
        'postal_zones',
        'parent_id',
        'type',
    ];

    /**
     * @var array $casts
     */
    protected $casts = [
        'postal_zones' => 'array'
    ];

    /**
     * Extend Model booting
     */
    public static function boot()
    {
        parent::boot();

        static::deleting(function (self $zone) {
            //Delete subzones related items
            $zone->subZones
                ->each(function ($subzone) {
                    $subzone->countries()->delete();
                    $subzone->rates()->delete();
                });

            //Delete zone related items
            $zone->countries()->sync([]);
            $zone->rates()->delete();
            $zone->subzones()->delete();
        });
    }

    /**
     * Relationship with Countries that have this zone
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function countries()
    {
        return $this->belongsToMany(Country::class, 'shipping_zone_countries', 'zone_id', 'country_id');
    }

    /**
     * Relationships of rates in this zone
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function rates()
    {
        return $this->hasMany(TableRate::class, 'zone_id');
    }

    /**
     * Relationships with sub zones
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function subZones()
    {
        return $this->hasMany(self::class, 'parent_id');
    }

    /**
     * Relationship with postal sub-zones
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function postalZones()
    {
        return $this->hasMany(static::class, 'parent_id', 'id')
            ->with('rates');
    }

    /**
     * Find a zones by country code
     *
     * @param string $country_code
     * @return \Illuminate\Database\Eloquent\Builder[]|\Illuminate\Database\Eloquent\Collection
     */
    public static function zonesForCountry($country_code = "GB")
    {
        return static::query()
            ->where('disabled', 0)
            ->whereHas('countries', function ($query) use ($country_code) {
                $query->where('code', $country_code);
            })
            ->whereDoesntHave('subZones.countries', function ($query) use ($country_code) {
                $query->where('code', $country_code);
            })
            ->get();
    }

    /**
     * Scope a query to only include zones with parent_id = 0.
     *
     * @param Builder $query
     * @return mixed
     */
    public function scopeAllMainZones(Builder $query)
    {
        return $query->where('parent_id', 0);
    }

    /**
     * Scope a query to only include zones with parent_id = 0 with their sub zones.
     *
     * @param Builder $query
     * @return mixed
     */
    public function scopeAllMainZonesWithSubZones(Builder $query)
    {
        return $query->with('postalZones')->allMainZones();
    }

    /**
     * Scope a query to only include enabled zones
     *
     * @param Builder $query
     * @param int $disabled
     * @return mixed
     */
    public function scopeEnabled(Builder $query, $disabled = 0)
    {
        return $query->where('disabled', $disabled);
    }

    /**
     * Get unique countries (name, code) from enabled zones
     *
     * @return \Illuminate\Support\Collection
     */
    public static function getUniqueCountriesCodeNameFromEnabled()
    {
        $countries = self::query()
            ->enabled()
            ->with('countries')
            ->get()
            ->map(function ($zone) {
                return $zone->countries->toArray();
            })
            ->toArray();


        if (!empty($countries)) {
            $countries = collect(array_merge(...$countries))
                ->sortBy('name')
                ->pluck('name', 'code');
        }

        return $countries;
    }
}
