<?php

namespace Mtc\ShippingManager\DeliveryTypes;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Mtc\Basket\Contracts\BasketRepositoryInterface;
use Mtc\ShippingManager\Contracts\DeliveryMethodContract;
use Mtc\ShippingManager\Facades\ShippingModifiers;
use Mtc\ShippingManager\TableRate;

/**
 * Class TableRateDelivery
 *
 * @package Mtc\ShippingManager
 */
class TableRateDelivery implements DeliveryMethodContract
{
    /**
     * Retrieve methods that are applicable to basket
     *
     * @param BasketRepositoryInterface $basket
     */
    public function methodsForBasket(BasketRepositoryInterface $basket)
    {
        $restricted_methods = $this->getRestrictedMethods($basket->getItems());

        if (empty($basket->getDestinationAddress()->country) || $basket->needsDelivery() === false) {
            return [];
        }

        $rates = TableRate::search(
            $basket->getDestinationAddress(),
            $basket->getWeight(),
            $basket->getCostSubtotalAttribute()
        );

        if (!empty($restricted_methods)) {
            $rates = $rates->reject(fn($rate) => in_array($rate->id, $restricted_methods));
        }
        $rates = ShippingModifiers::apply($basket, $rates);
        $this->setActiveMethod($basket, $rates);
        return App::make('delivery_method_response_class')->setResource($rates);
    }

    /**
     * Set active delivery method
     *
     * @param BasketRepositoryInterface $basket
     * @param Collection $rates
     */
    public function setActiveMethod(BasketRepositoryInterface $basket, Collection $rates)
    {
        if ($this->hasInvalidDeliverySurcharge($basket, $rates)) {
            if ($rates->count() == 0) {
                $basket->removeSurchargeByType('delivery');
                return;
            }

            $basket->updateSurchargeByType('delivery', $rates->first()->id);
            $basket->getModel()
                ->update([
                    'shipping_type' => 'delivery'
                ]);
        }

        $this->setActiveRateInList($basket, $rates);
    }

    public function hasMethods(): bool
    {
        return TableRate::query()->exists();
    }

    /**
     * Check if the delivery surcharge applied to basket matches the rate list
     *
     * @param BasketRepositoryInterface $basket
     * @param Collection $rates
     * @return bool
     */
    protected function hasInvalidDeliverySurcharge(BasketRepositoryInterface $basket, Collection $rates)
    {
        if (!$basket->hasSurcharge(0, 'delivery')) {
            return true;
        }

        return $rates->where('id', $basket->getSurchargeIdFromType('delivery'))->count() == 0;
    }

    /**
     * Check and set the active rate as active in the list based on basket
     *
     * @param BasketRepositoryInterface $basket
     * @param $rates
     */
    protected function setActiveRateInList(BasketRepositoryInterface $basket, $rates)
    {
        $rates->each(function (TableRate $rate) use ($basket) {
            if ($basket->hasSurcharge($rate->id, 'delivery')) {
                $rate->active = true;
            }
        });
    }

    /**
     * @param $basket_items
     * @return array
     */
    private function getRestrictedMethods($basket_items): array
    {
        $items = collect($basket_items);
        return collect(config('shipping_manager.restricted_items_for_methods'))
            ->filter(
                fn($restriction) => collect($restriction['purchasable_types'])->intersect(
                    $items->pluck('purchasable_type')
                )
            )
            ->pluck('method_ids')
            ->flatten(1)
            ->toArray();
    }
}
