<?php

namespace Mtc\ShippingManager\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\ShippingManager\Country;
use Mtc\ShippingManager\Zone;

/**
 * Class ZoneController
 *
 * @package Mtc\ShippingManager
 */
class ZoneController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return string
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Syntax
     */
    public function index()
    {

        $self_action = url('/admin/shipping-manager/zones');

        $shipping_zones = Zone::query()->allMainZonesWithSubZones()->get();

        $this->page_meta['page_name'] = 'Shipping Zones';

        return template('ShippingManager/admin/zones/list.twig', [
            'page_meta' => $this->page_meta,
            'self_action' => $self_action,
            'shipping_zones' => $shipping_zones,
            'selected_tab_nav' => 'list',
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return string
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Syntax
     */
    public function create()
    {
        $store_action = url('/admin/shipping-manager/zones');

        $shipping_zones = Zone::query()
            ->allMainZones()
            ->pluck('name', 'id');

        $shipping_zones->prepend('No Parent', 0);

        $this->page_meta['page_name'] = 'Add Shipping Zone';

        return template('ShippingManager/admin/zones/edit.twig', [
            'page_meta' => $this->page_meta,
            'selected_tab_nav' => 'add_new',
            'shipping_zones' => $shipping_zones,
            'store_action' => $store_action,
            'default_type' => Zone::DEFAULT_TYPE,
            'zone_types' => Zone::TYPES,
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param Request $request
     * @param $shipping_zone_id
     * @return string
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Syntax
     */
    public function edit(Request $request, $shipping_zone_id)
    {
        $update_action = dirname($request->getPathInfo());

        $shipping_zone = Zone::find($shipping_zone_id);

        $shipping_zones = Zone::query()
            ->allMainZones()
            ->where('id', '!=', $shipping_zone_id)
            ->pluck('name', 'id');

        $shipping_zones->prepend('No Parent', 0);

        $this->page_meta['page_name'] = 'Shipping Zone';

        $countries = Country::all()->groupBy('continent_name');

        return template('ShippingManager/admin/zones/edit.twig', [
            'page_meta' => $this->page_meta,
            'shipping_zone' => $shipping_zone,
            'shipping_zones' => $shipping_zones,
            'update_action' => $update_action,
            'countries' => $countries,
            'selected_tab_nav' => 'edit',
            'default_type' => Zone::DEFAULT_TYPE,
            'zone_types' => Zone::TYPES,
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param Request $request
     * @param $collect_location_id
     * @return mixed
     */
    public function update(Request $request, $shipping_zone_id)
    {
        $validated = $this->validate($request, [
            'name' => 'required',
            'type' => 'required',
            'parent_id' => 'required|numeric'
        ]);

        $shipping_zone = Zone::query()->findOrFail($shipping_zone_id);
        if ($request->has('postal_zones')) {
            $shipping_zone->postal_zones = $request->input('postal_zones');
        }

        $shipping_zone->fill($request->all())->save();

        if ($request->has('countries')) {
            $shipping_zone->countries()->sync($request->input('countries'));
        }


        session()->flash('success', 'Shipping Zone updated');
        return redirect()->back();
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param Request $request
     * @return mixed
     */
    public function store(Request $request)
    {
        $validated = $this->validate($request, [
            'name' => 'required',
            'type' => 'required',
            'parent_id' => 'required|numeric'
        ]);

        $shipping_zone = Zone::query()
            ->create([
                'name' => $validated['name'],
                'type' =>  Zone::TYPES[$validated['type']],
                'parent_id' => $validated['parent_id'],
            ]);

        session()->flash('success', 'Shipping Zone created');
        return redirect('/admin/shipping-manager/zones/' . $shipping_zone->id . '/edit');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param $zone_id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy($zone_id)
    {
        $zone = Zone::query()->findOrFail($zone_id);
        $zone->delete();

        session()->flash('success', 'Zone deleted');
        return redirect()->back();
    }
}
