<?php

namespace Mtc\ShopDashboardStats\Http\Controllers;

use App\Http\Controllers\Controller;
use Carbon\Carbon;
use Mtc\AnalyticsStats\Stats;
use Mtc\ShopDashboardStats\Http\Requests\StatsRequest;
use Mtc\ShopDashboardStats\OrderStats;

class DashboardStatsController extends Controller
{
    /**
     * Weekly sales stats
     *
     * @return array
     */
    public function weekly()
    {
        $now = Carbon::now();
        $data = collect([]);

        for ($i = 0; $i < 7; $i++) {
            $start = $now->copy()->subDays($i)->startOfDay();
            $end = $now->copy()->subDays($i)->endOfDay();

            $data->prepend([
                $start->format('M jS'),
                (float)OrderStats::totalSales($start, $end),
                (float)OrderStats::totalSales($start->subYear(), $end->subYear()),
            ]);
        }

        $data->prepend(['Day', 'Sales (GBP)', 'Last Year']);
        return $this->response([
            'data' => $data
        ]);
    }

    /**
     * Monthly sales stats
     *
     * @return array
     */
    public function monthly()
    {
        $now = Carbon::now();
        $data = collect([]);

        for ($i = 0; $i < 7; $i++) {
            $start = $now->copy()->startOfMonth()->subMonths($i);
            $end = $now->copy()->startOfMonth()->subMonths($i)->endOfMonth();

            $data->prepend([
                $start->format('M'),
                (float)OrderStats::totalSales($start, $end),
                (float)OrderStats::totalSales($start->subYear(), $end->subYear()),
            ]);
        }

        $data->prepend([ 'Month', 'Sales (GBP)', 'Last Year' ]);
        return $this->response([
            'data' => $data
        ]);
    }

    /**
     * @param StatsRequest $request
     * @return array
     */
    public function sales(StatsRequest $request)
    {
        $orders = OrderStats::totalSales($request->start_date, $request->end_date);
        return $this->response([
            'html' => view('mtc-shop-dashboard-stats::sales', [
                'order_value' => $orders,
                'average_order_value' => OrderStats::averageOrderValue($request->start_date, $request->end_date),
                'most_sold_items' => OrderStats::mostItemsSold($request->start_date, $request->end_date, 10)
            ])->render(),
        ]);
    }

    /**
     * @param StatsRequest $request
     * @return array
     */
    public function orders(StatsRequest $request)
    {
        $order_count = OrderStats::orderCount($request->start_date, $request->end_date);
        $item_count = OrderStats::itemCount($request->start_date, $request->end_date);
        $difference_in_days = $request->start_date->diffInDays($request->end_date) ?: 1;

        $num_ppc_orders = 0;
        return $this->response([
            'html' => view('mtc-shop-dashboard-stats::orders', [
                'order_count' => $order_count,
                'item_count' => $item_count,
                'avg_items_per_order' => $item_count / ($order_count ?: 1),
                'avg_orders_per_day' => $order_count / $difference_in_days,
                'num_ppc_orders' => $num_ppc_orders,
            ])->render(),
        ]);
    }

    public function analytics(StatsRequest $request)
    {
        $view_id = config('mtc-shop-dashboard-stats.analytics_view_id');
        if (empty($view_id)) {
            return $this->response([
                'html' => 'To view analytics, set mtc-shop-dashboard-stats.analytics_view_id or ENV GOOGLE_ANALYTICS_VIEW_ID',
            ]);
        }

        $analytics = new Stats($view_id);

        $data = collect([
            'Number of Visits' => $analytics->getTotalVisits($request->start_date, $request->end_date),
            'Number of Unique Visitors' => $analytics->getTotalVisitors($request->start_date, $request->end_date),
            'Conversion Rate %' => $analytics->getConversionRate($request->start_date, $request->end_date),
            'Visitors to Basket Page' => $analytics->getTotalVisitorsToPage($request->start_date, $request->end_date, route('basket.index', [], false)),
        ])->filter(function($row) {
            return $row > 0;
        });

        return $this->response([
            'html' => view('mtc-shop-dashboard-stats::analytics', compact('data'))->render(),
        ]);
    }

    /**
     * The admin.js in Core expects everything to be wrapped up in a 'data' attribute.
     * Instead of editing each response to add this, I'm just adding here instead.
     * 
     * @return Illuminate\Http\Response
     */
    private function response($data)
    {
        return response()
            ->json(compact('data'));

    }
}
