<?php

namespace Mtc\SiteNavigation;

use Illuminate\Support\Facades\Cache;

/**
 * Class Navigation
 *
 * @package Mtc\SiteNavigation
 */
class Navigation
{
    /**
     * Show navigation menu.
     * Determines cache level used
     *
     * @param $menu_slug
     * @param array $options
     * @return mixed|string
     */
    public function show($menu_slug, $options = [])
    {
        if (config('site_navigation.use_html_cache') === true) {
            $cache_expire = now()->addMinutes(config('site_navigation.cache_lifetime_minutes'));
            return Cache::remember("navigation-menu-html-{$menu_slug}", $cache_expire, function () use ($menu_slug, $options) {
                $this->render($menu_slug, $options);
            });
        }

        return $this->render($menu_slug, $options);
    }

    /**
     * Render menu for site usage
     *
     * @param string $menu
     * @param array $options
     * @return string
     */
    protected function render($menu_slug, $options = [])
    {
        $template_location = $this->getTemplate($options);

        // fallback to container class defined in settings
        if (!isset($options['first_container_class'])) {
            $options['first_container_class'] = config('site_navigation.first_container_class', '');
        }

        $menu = SiteNavigation::find($menu_slug);
        if (!$menu || $menu->published === false) {
            return '';
        }

        $cache_expire = now()->addMinutes(config('site_navigation.cache_lifetime_minutes'));
        $menu_tree = Cache::remember("navigation-menu-{$menu_slug}", $cache_expire, function () use ($menu) {
            // TODO: category / brand / cms list injection
            return SiteNavigation::published()->descendantsOf($menu->id)->toTree($menu->id);
        });

        $options['container_class'] = $menu->css_class;

        return template($template_location, [
            'items' => $menu_tree,
            'options' => $options,
            'top_level' => true,
        ]);
    }

    /**
     * Get the template for the menu
     *
     * @param $options
     * @return string
     */
    private function getTemplate($options)
    {
        //Check if mega-menu option enabled
        if (isset($options['mega_menu']) && $options['mega_menu'] === true) {
            return 'site_navigation/mega_menu.twig';
        }

        //Default template menu
        return 'site_navigation/menu.twig';
    }
}
