<?php

namespace Mtc\SquarePay\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\URL;
use Mtc\Basket\Contracts\BasketRepositoryInterface;
use Mtc\Checkout\Contracts\InvoiceFactoryContract;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\PayableFactoryContract;
use Mtc\Foundation\Address;
use Mtc\ShippingManager\Facades\DeliveryMethod;
use Mtc\SquarePay\Http\Requests\SetAddress;
use Ramsey\Uuid\Uuid;
use Square\Environment;
use Square\Models\CreatePaymentRequest;
use Square\Models\CreatePaymentResponse;
use Square\Models\Money;
use Square\SquareClient;

class SquareController
{
    /**
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return array|\Illuminate\Contracts\Routing\ResponseFactory|\Illuminate\Http\Response
     */
    public function index(Request $request, InvoiceRepositoryContract $invoice)
    {
        $invoice->load($request->input('invoice_id'));
        $apiResponse = $this->makePaymentRequest($request, $invoice);

        if ($apiResponse->isSuccess()) {
            /** @var CreatePaymentResponse $createPaymentResponse */
            $createPaymentResponse = $apiResponse->getResult();
            if ($createPaymentResponse->getPayment()->getStatus() === 'COMPLETED') {
                $invoice->savePayment([
                    'provider' => 'square',
                    'amount' => $createPaymentResponse->getPayment()->getAmountMoney()->getAmount(),
                    'reference' => $createPaymentResponse->getPayment()->getId(),
                    'details' => $createPaymentResponse,
                    'confirmed_at' => now(),
                ]);

                return [
                    'success' => true,
                    'redirect' => URL::signedRoute(
                        'successful_payment',
                        ['id' => $invoice->getId() ],
                        null,
                        config('app.env') === 'production'
                    ),
                ];
            }
        }
        return response($apiResponse->getErrors(), 422);
    }

    public function setShippingAddress(SetAddress $request, BasketRepositoryInterface $basket)
    {
        $basket->setAddress(\Mtc\Basket\Address::TYPE_SHIPPING, $request->only(array_keys(Address::$blueprint)));
        return [
            'methods' => DeliveryMethod::methodsForBasket($basket),
            'basket' => App::make(config('basket.checkout_ajax_response_class')),
        ];
    }

    /**
     * Make payment request
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return \Square\Http\ApiResponse
     * @throws \Square\Exceptions\ApiException
     */
    protected function makePaymentRequest(Request $request, InvoiceRepositoryContract $invoice)
    {
        $client = new SquareClient([
            'accessToken' => config('square.access_token'),
            'environment' => config('square.live') ? Environment::PRODUCTION : Environment::SANDBOX,
        ]);
        $moneyAmount = new Money();
        $moneyAmount->setAmount($invoice->getOutstandingAmountInCurrency() * 100);
        $moneyAmount->setCurrency($invoice->getCurrency());

        $body = new CreatePaymentRequest(
            $request->input('token'),
            Uuid::uuid4(),
            $moneyAmount
        );
        if ($request->filled('verify_token') && $request->input('verify_token') !== 'undefined') {
            $body->setVerificationToken($request->input('verify_token'));
        }
        $body->setAutocomplete(true);
        $body->setLocationId(config('square.location_id'));
        $body->setReferenceId($invoice->getReference());
        $body->setNote('Website Order ' . $invoice->getReference());

        return $client->getPaymentsApi()->createPayment($body);
    }
}
