<?php

namespace Mtc\SquarePay;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\PayableContract;
use Mtc\Checkout\Contracts\PaymentGateway;
use Mtc\Checkout\PaymentForm;

class Square implements PaymentGateway
{
    /**
     * Check if the gateway is available for use on this payment.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return bool
     */
    public function isApplicable(InvoiceRepositoryContract $invoice, $payable): bool
    {
        return App::make(config('square.applicable_check_class'))->handle($invoice, $payable);
    }

    /**
     * Render the payment template.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return string
     */
    public function getPaymentForm(InvoiceRepositoryContract $invoice): PaymentForm
    {
        $recipient = $invoice->getModel()->recipient;
        return new PaymentForm('square-payment', 'vue-component', [
            'app_id' => config('square.app_id'),
            'location_id' => config('square.location_id'),
            'invoice_id' => $invoice->getId(),
            'name' => __('square::square.payment_option_name'),
            'test_mode' => config('square.live') !== true,
            'verify_details' => [
                'amount' => (string)$invoice->getOutstandingAmountInCurrency(),
                'currencyCode' => $invoice->getCurrency(),
                'intent' => 'CHARGE',
                'billingContact' => [
                    'addressLines' => collect([ $recipient->address1, $recipient->address2])->filter(),
                    'familyName' => $recipient->last_name,
                    'givenName' => $recipient->first_name,
                    'email' => $invoice->getEmail(),
                    'country' => $recipient->country,
                    'phone' => $invoice->getModel()->contact_number,
                    'region' => $recipient->county ?? '',
                    'city' => $recipient->city,
                ]
            ]
        ]);
    }

    /**
     * Charge payment on invoice
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return bool
     * @throws \Exception
     */
    public function charge(Request $request, InvoiceRepositoryContract $invoice): array
    {
        return [];
    }

}