<?php

namespace Mtc\SquarePay\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Validator;
use Mtc\Foundation\Rules\ValidPostcode;

class SetAddress extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        return [
            'city' => 'required',
            'country' => 'required|exists:countries,code',
            'postcode' => new ValidPostcode($this->request->get('country')),
        ];
    }

    /**
     * Extend the validation data to pre-fill with missing data
     *
     * @return array
     */
    public function validationData()
    {
        $this->preFillMissing();
        return parent::validationData();
    }

    /**
     * We receive only partial address here, also in different format.
     * Adjustment to make it uniform with DB format
     */
    protected function preFillMissing()
    {
        $fields = collect([
                'county' => $this->input('state'),
                'country' => $this->input('countryCode'),
                'postcode' => $this->ensureValidPostcode(),
                'first_name' => '',
                'last_name' => '',
                'address1' => '',
            ])
            ->reject(function ($value, $field) {
                $this->exists($field);
            });

        $this->merge($fields->toArray());
    }

    /**
     * Request will give you only partial UK postcode.
     * This will fail validation so we need to fake the second half
     *
     * @return string
     */
    protected function ensureValidPostcode()
    {
        $validator = Validator::make($this->all(), [
            'postalCode' => new ValidPostcode($this->request->get('countryCode')),
        ]);

        if ($validator->fails() && $this->input('countryCode') === 'GB') {
            return $this->input('postalCode') . '1xx';
        }

        return $this->input('postalCode');
    }

}
