<?php

namespace Mtc\Stripe;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\Basket\Contracts\BasketRepositoryInterface;
use Mtc\Checkout\Contracts\ExternalPaymentGateway;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\PayableContract;
use Mtc\Checkout\PaymentForm;

/**
 * Stripe Payment Request extension to support apple / google pay
 *
 * @package  Mtc\Stripe
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class StripePaymentRequest implements ExternalPaymentGateway
{
    public function __construct()
    {
        if (config('stripe.account_id')) {
            \Stripe\Stripe::setAccountId(config('stripe.account_id'));
        }
    }

    /**
     * Check if the gateway is available for use on this payment.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return bool
     */
    public function isApplicable(BasketRepositoryInterface $basket): bool
    {
        if ($basket->getCostTotalAttribute() <= config('stripe.minimal_transaction_amount')) {
            return false;
        }

        return App::make(config('stripe.external_payments_applicable_check_class'))->handle($basket);
    }

    /**
     * Render the payment template.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return string
     */
    public function getPaymentForm(BasketRepositoryInterface $basket): PaymentForm
    {
        \Stripe\Stripe::setApiKey(config('stripe.private_key'));
        return new PaymentForm('stripe-payment-request', 'vue-component', [
            'stripe_public_key' => config('stripe.public_key'),
            'stripe_account_id' => config('stripe.account_id'),
        ]);
    }

    /**
     * Charge payment on invoice
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return bool
     * @throws \Exception
     */
    public function charge(Request $request, InvoiceRepositoryContract $invoice): array
    {
        // Payment flow is handled in StripePaymentRequestController
        return [];
    }
}
