<?php

namespace Mtc\Stripe\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Validator;
use Mtc\Foundation\Rules\ValidPostcode;

/**
 * Class SetAddress
 *
 * @package Mtc\Stripe\Http\Requests
 */
class SetAddress extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        return [
            'city' => 'required',
            'country' => 'required|exists:countries,code',
            'postcode' => new ValidPostcode($this->request->get('country')),
        ];
    }

    /**
     * Extend the validation data to pre-fill with missing data
     *
     * @return array
     */
    public function validationData()
    {
        $this->preFillMissing();
        return parent::validationData();
    }

    /**
     * Stripes Apple pay gives you only partial information on address change.
     * This will fail validation so we need to fake the second half
     */
    protected function preFillMissing()
    {
        $fields = collect([
                'postcode' => $this->ensureValidPostcode(),
                'first_name' => '',
                'last_name' => '',
                'address1' => '',
            ])
            ->reject(function ($value, $field) {
                $this->exists($field);
            });

        $this->merge($fields->toArray());
    }

    /**
     * Stripes Apple pay gives you only partial UK postcode.
     * This will fail validation so we need to fake the second half
     *
     * @return string
     */
    protected function ensureValidPostcode()
    {
        $validator = Validator::make($this->all(), [
            'postalCode' => new ValidPostcode($this->request->get('country')),
        ]);

        if ($validator->fails() && $this->request->get('country') === 'GB') {
            return $this->input('postalCode') . '1xx';
        }

        return $this->input('postalCode');
    }
}
